<#
.SYNOPSIS
  Powershell script to get information from microsoft endpoint manager and represent in prtg
.NOTES
  Version:        1.0
  Author:         roman huesler / opensight.ch
  Creation Date:  17.03.2023
  Purpose/Change: Initial script development 
.EXAMPLE
  V1 Powershell Script - v1_prtg_intune_app_status.ps1 
                         -app_id <azure-app-id> 
                         -tenant_id <azure-tenant-id> 
                         -client_secret <azure-client-secret>
                         -states <states-to-show-in-prtg>
                         -softwarenames <intune-app-names-to-show-in-prtg>
#>

# script parameters
param(
    [string]$app_id,
    [String]$tenant_id,
    [String]$client_secret,
    [String]$states = "total;all;installed;failed;notInstalled;uninstallFailed;pendingInstall;unknown;notApplicable",
    [String]$softwarenames = "overall;all",
    [bool]  $debug = $false
)

# possible states:
# installed, failed, notInstalled, uninstallFailed, pendingInstall, unknown, notApplicable.
$possiblestates = @("installed", "failed", "notInstalled", "uninstallFailed", "pendingInstall", "unknown", "notApplicable")

#############################################################################
# simple logging function
#############################################################################
function log($text) {
    if($debug -eq $false) { return }
    $dt = get-date -format "dd.MM.yyyy HH:mm:ss"
    write-host $dt - $text
}

# microsoft authentication library module import
log "importing module msal.ps"
Import-Module MSAL.PS

#############################################################################
# If Powershell is running the 32-bit version on a 64-bit machine, we 
# need to force powershell to run in 64-bit mode .
# http://cosmonautdreams.com/2013/09/03/Getting-Powershell-to-run-in-64-bit.html
#############################################################################
if ($env:PROCESSOR_ARCHITEW6432 -eq "AMD64") {
    log "Y'arg Matey, we're off to 64-bit land....."
    if ($myInvocation.Line) {
        &"$env:WINDIR\sysnative\windowspowershell\v1.0\powershell.exe" -NonInteractive -NoProfile $myInvocation.Line
    }else{
        &"$env:WINDIR\sysnative\windowspowershell\v1.0\powershell.exe" -NonInteractive -NoProfile -file "$($myInvocation.InvocationName)" $args
    }
    exit $lastexitcode
}
 
# acquire azure authentication token
log "acquire azure authentication token"
log "tenant_id: $($tenant_id)"
log "app_id: $($app_id)"
log "client_secret: $($client_secret)"
$MsalToken = Get-MsalToken -TenantId $tenant_id -ClientId $app_id -ClientSecret ($client_secret | ConvertTo-SecureString -AsPlainText -Force)
log "got msalToken: $($msalToken)"
 
#Connect to Graph using access token
log "connecting to msgraph api with token"
Connect-Graph -AccessToken $MsalToken.AccessToken | out-null

#############################################################################
# main function to query intune (graph api)
#############################################################################
function getIntune($token,  $endpoint, $graphApiVersion = "v1.0") {
    try {
        $uri = "https://graph.microsoft.com/$graphApiVersion/$($endpoint)"
        log "query graph api: $($uri)"
        log "token-length: $($token.length)"

        $headers = @{
            Authorization="Bearer $token"
        }
        $GraphResponse = (Invoke-RestMethod -Uri $uri -Headers $headers -Method Get)
        
        $collection = @()

        $devices = $GraphResponse.value
        $collection += $devices
        
        $DevicesNextLink = $GraphResponse."@odata.nextLink"
        while ($DevicesNextLink -ne $null){
            $GraphResponse = (Invoke-RestMethod -Uri $DevicesNextLink -Headers $headers -Method Get)
            $DevicesNextLink = $GraphResponse."@odata.nextLink"
            $collection += $GraphResponse.value
        }
        log "got $(($collection | measure).Count) results from graph request"
        return $collection
    } catch {
        $ex = $_.Exception
        $errorResponse = $ex.Response.GetResponseStream()
        $reader = New-Object System.IO.StreamReader($errorResponse)
        $reader.BaseStream.Position = 0
        $reader.DiscardBufferedData()
        $responseBody = $reader.ReadToEnd();
        log "Response content:`n$responseBody"
        log "Request to $Uri failed with HTTP Status $($ex.Response.StatusCode) $($ex.Response.StatusDescription)"
        Write-Host "Response content:`n$responseBody" -f Red
        Write-Error "Request to $Uri failed with HTTP Status $($ex.Response.StatusCode) $($ex.Response.StatusDescription)"
        throw "Get-IntuneManagedDevices error"
    }
}

#############################################################################
# get a list of all app device states as array with correct format
#############################################################################
function getAllAppDeviceStatuses($token) {
    # get all mobile apps from intune
    log "get intune mobile apps as formatted array"
    $apps = getIntune -token $MsalToken.AccessToken -endpoint "deviceAppManagement/mobileApps" -graphApiVersion "beta"
    log "found $(($apps | measure).Count) apps"
    # filter out apps that are actually assigned
    log "filtering apps that are actually assigned"
    $apps = $apps | Where-Object { $_.isAssigned -eq $true}
    log "now got $(($apps | measure).Count) apps"
    # prepare output array
    log "prepare output array"
    $AppInstallationState = @()
    # fetch app install states
    foreach($app in $apps) {
        # query graph api to get appstate for this app
        log "querying graph api to get devicestates for app $($app.displayName) ($($app.id))"
        $AllAppStates = getIntune -token $MsalToken.AccessToken -endpoint "deviceAppManagement/mobileApps/$($app.id)/deviceStatuses" -graphApiVersion "beta"
        log "num results: $(($AllAppStates | measure).Count)"

        if( $AllAppStates.Count -eq 0 )
        {  
            # App is not installed on any device
            log "return default empty appstate as there were no appstates retrieved from graphApi for this app"
            $AppStateObj = New-Object -TypeName PSObject
            $AppStateObj | Add-Member -MemberType NoteProperty -Name AppID -Value $app.id
            $AppStateObj | Add-Member -MemberType NoteProperty -Name Displayname -Value $app.displayName
            $AppStateObj | Add-Member -MemberType NoteProperty -Name AppVersion -value $app.displayVersion
            $AppStateObj | Add-Member -MemberType NoteProperty -Name isAssigned -value $app.isAssigned
            $AppStateObj | Add-Member -MemberType NoteProperty -Name DeviceName -Value $null
            $AppStateObj | Add-Member -MemberType NoteProperty -Name DeviceID -Value $null
            $AppStateObj | Add-Member -MemberType NoteProperty -Name LastDeviceSync -Value $null
            $AppStateObj | Add-Member -MemberType NoteProperty -Name InstallState -value $null
            $AppInstallationState += $AppStateObj
        } else {   
            # App is installed on multiple devices
            foreach($AppState in $AllAppStates)
            {   
                log "add appstate to array for app $($app.id) and devic $($AppState.deviceId)"
                $AppStateObj = New-Object -TypeName PSObject
                $AppStateObj | Add-Member -MemberType NoteProperty -Name AppID -Value $app.id
                $AppStateObj | Add-Member -MemberType NoteProperty -Name Displayname -Value $app.displayName
                $AppStateObj | Add-Member -MemberType NoteProperty -Name AppVersion -value $app.displayVersion
                $AppStateObj | Add-Member -MemberType NoteProperty -Name isAssigned -value $app.isAssigned
                $AppStateObj | Add-Member -MemberType NoteProperty -Name DeviceName -Value $AppState.deviceName
                $AppStateObj | Add-Member -MemberType NoteProperty -Name DeviceID -Value $AppState.deviceId
                $AppStateObj | Add-Member -MemberType NoteProperty -Name LastDeviceSync -Value (get-date $AppState.lastSyncDateTime -Format G)
                $AppStateObj | Add-Member -MemberType NoteProperty -Name InstallState -value $AppState.installState
                $AppInstallationState += $AppStateObj

            }

        }
        $Count++ 
    }
    log "returning now $(($AppInstallationState | measure).Count) app states"
    return $AppInstallationState
}

#############################################################################
# MAIN SCRIPT
#############################################################################
log "================================"
log "PRTG - Intune App States V1.0   "
log "================================"

# run function to get all appstates formatted as array
log "getting install states"
$installStates = getAllAppDeviceStatuses -token $MsalToken.AccessToken

# get unique app names
log "filtering out unique app names"
$appNames = $installStates | Select-Object -Unique -Property Displayname
log "found $(($appNames | measure).Count) unique app names"
log "states are $($states)"

log "start returning prtg output..."
"<prtg>"

# installstate overall
log "returning overall state"
if("overall" -in $softwarenames.Split(';')) { 
    $allstates = ($installStates | measure).Count
    "<result>"
    "<channel>all states - overall</channel>"
    "<value>$allstates</value>"
    "</result>"  
        
    foreach($possiblestate in $possiblestates) {
        if($possiblestate -notin $states.Split(';')) { continue }
        $res = ($installStates | Where-Object { $_.InstallState -eq $possiblestate } | measure).Count
        "<result>"
        "<channel>$($possiblestate) - overall</channel>"
        "<value>$($res)</value>"
        "</result>"
    }
}

# installstate by software
log "start returning installstates by software..."
foreach($app in $appNames) {
    $softwarename = $app.Displayname
    if($softwarename.toLower() -notin $softwarenames.toLower().Split(';') `
    -and "all" -notin $softwarenames.toLower().Split(';')) { continue }

    if("total" -in $states.Split(';')) { 
        $allstates = ($installStates | Where-Object { $_.Displayname -eq $app.Displayname } | measure).Count
        "<result>"
        "<channel>total - $($app.Displayname)</channel>"
        "<value>$allstates</value>"
        "</result>"   
    }
    foreach($possiblestate in $possiblestates) {
        if($possiblestate -notin ($states -split ';') -and "all" -notin ($states -split ';')) {
            continue 
        }
        $res = ($installStates | Where-Object { $_.Displayname -eq $app.Displayname -and $_.InstallState -eq $possiblestate } | measure).Count
        "<result>"
        "<channel>$($possiblestate) - $($app.Displayname)</channel>"
        "<value>$($res)</value>"
        "</result>"
    }
}
"</prtg>"