<#
.SYNOPSIS
  Powershell script to get information from microsoft endpoint manager and represent in prtg
.NOTES
  Version:        2.0
  Author:         roman huesler / opensight.ch
  Creation Date:  16.10.2022
  Purpose/Change: Initial script development 
.EXAMPLE
  V1 Powershell Script - v1_prtg_intune_device_monitor.ps1 -username -password
  V2 Powershell Script - v2_prtg_intune_device_monitor.ps1 -app_id xxxx -tenant_id xxxx -client_secret xxxx
#>

#authentication
param(
    [string]$app_id,
    [String]$tenant_id,
    [String]$client_secret
)

# microsoft authentication library module import
Import-Module MSAL.PS

#############################################################################
#If Powershell is running the 32-bit version on a 64-bit machine, we 
#need to force powershell to run in 64-bit mode .
#http://cosmonautdreams.com/2013/09/03/Getting-Powershell-to-run-in-64-bit.html
#############################################################################
if ($env:PROCESSOR_ARCHITEW6432 -eq "AMD64") {
    write-warning "Y'arg Matey, we're off to 64-bit land....."
    if ($myInvocation.Line) {
        &"$env:WINDIR\sysnative\windowspowershell\v1.0\powershell.exe" -NonInteractive -NoProfile $myInvocation.Line
    }else{
        &"$env:WINDIR\sysnative\windowspowershell\v1.0\powershell.exe" -NonInteractive -NoProfile -file "$($myInvocation.InvocationName)" $args
    }
exit $lastexitcode
}
 
# acquire azure authentication token
$MsalToken = Get-MsalToken -TenantId $tenant_id -ClientId $app_id -ClientSecret ($client_secret | ConvertTo-SecureString -AsPlainText -Force)
 
#Connect to Graph using access token
Connect-Graph -AccessToken $MsalToken.AccessToken | out-null

#############################################################################
# main function to query all intune devices (graph api)
#############################################################################
function getIntuneDevices($token) {
    $graphApiVersion = "v1.0"

    try {
        $Resource = "deviceManagement/managedDevices"
        $uri = "https://graph.microsoft.com/$graphApiVersion/$($Resource)"

        $headers = @{
            Authorization="Bearer $token"
        }
        $DevicesResponse = (Invoke-RestMethod -Uri $uri -Headers $headers -Method Get)
        
        $collection = @()

        $devices = $DevicesResponse.value
        $collection += $devices
        
        $DevicesNextLink = $DevicesResponse."@odata.nextLink"
        while ($DevicesNextLink -ne $null){
            $DevicesResponse = (Invoke-RestMethod -Uri $DevicesNextLink -Headers $headers -Method Get)
            $DevicesNextLink = $DevicesResponse."@odata.nextLink"
            $collection += $DevicesResponse.value
        }
        return $collection
    } catch {
        $ex = $_.Exception
        $errorResponse = $ex.Response.GetResponseStream()
        $reader = New-Object System.IO.StreamReader($errorResponse)
        $reader.BaseStream.Position = 0
        $reader.DiscardBufferedData()
        $responseBody = $reader.ReadToEnd();
        Write-Host "Response content:`n$responseBody" -f Red
        Write-Error "Request to $Uri failed with HTTP Status $($ex.Response.StatusCode) $($ex.Response.StatusDescription)"
        throw "Get-IntuneManagedDevices error"
    }
}

# gather device information from intune
$devices = getIntuneDevices -token $MsalToken.AccessToken

# take counts
$all_devices = ($devices | measure).Count
$os_windows = ($devices | where-object { $_.operatingSystem -eq "Windows" } | measure).Count
$os_macos = ($devices | where-object { $_.operatingSystem -eq "MacOS" } | measure).Count
$os_android = ($devices | where-object { $_.operatingSystem -eq "Android" } | measure).Count
$os_ios = ($devices | where-object { $_.operatingSystem -eq "iOS" } | measure).Count
$os_unknown = ($devices | where-object { $_.operatingSystem -eq "" } | measure).Count
$compliant = ($devices | where-object { $_.complianceState -ne "compliant" } | measure).Count
$encrypted = ($devices | where-object { $_.isEncrypted -ne "True" } | measure).Count

#returning PRTG XML data
#https://www.paessler.com/manuals/prtg/custom_sensors
"<prtg>"
"<result>"
"<channel>Total Devices</channel>"
"<value>$all_devices</value>"
"</result>"
"<result>"
"<channel>Windows</channel>"
"<value>$os_windows</value>"
"</result>"
"<result>"
"<channel>MacOS</channel>"
"<value>$os_macos</value>"
"</result>"
"<result>"
"<channel>Android</channel>"
"<value>$os_android</value>"
"</result>"
"<result>"
"<channel>iPhone</channel>"
"<value>$os_ios</value>"
"</result>"
"<result>"
"<channel>Unknown</channel>"
"<value>$os_unknown</value>"
"</result>"
"<result>"
"<channel>Not Compliant</channel>"
"<value>$compliant</value>"
"</result>"
"<result>"
"<channel>Not Encrypted</channel>"
"<value>$encrypted</value>"
"</result>"
"</prtg>"