<?php
/**
 * Plugin Name: Drew's Simple Internal Linking
 * Plugin URI: https://chapin.io/drews-simple-internal-linking-wordpress-plugin/
 * Description: A simple plugin to automatically create internal links throughout your WordPress site. <a href="admin.php?page=drews-simple-internal-linking">Go to Dashboard</a>
 * Version: 1.0.2
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Author: Drew Chapin
 * Author URI: https://chapin.io
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: drews-simple-internal-linking
 * 
 * @package DrewsSimpleInternalLinking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('DREWS_IL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DREWS_IL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('DREWS_IL_VERSION', '1.0.2');

class DrewsSimpleInternalLinking {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_drews_il_create_link', array($this, 'ajax_create_link'));
        add_action('wp_ajax_drews_il_remove_link', array($this, 'ajax_remove_link'));
        add_action('wp_ajax_drews_il_re_run_link', array($this, 'ajax_re_run_link'));
        add_action('wp_ajax_drews_il_get_progress', array($this, 'ajax_get_progress'));
        
        // Create database table on activation
        register_activation_hook(__FILE__, array($this, 'create_database_table'));
    }
    
    public function create_database_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'drews_internal_links';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            anchor_text varchar(255) NOT NULL,
            target_url varchar(500) NOT NULL,
            scope varchar(20) NOT NULL DEFAULT 'both',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            last_run datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'themes.php',
            'Drew\'s Simple Internal Linking',
            'Internal Linking',
            'manage_options',
            'drews-simple-internal-linking',
            array($this, 'admin_page')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'appearance_page_drews-simple-internal-linking') {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('drews-il-admin', DREWS_IL_PLUGIN_URL . 'js/admin.js', array('jquery'), DREWS_IL_VERSION, true);
        wp_enqueue_style('drews-il-admin', DREWS_IL_PLUGIN_URL . 'css/admin.css', array(), DREWS_IL_VERSION);
        
        wp_localize_script('drews-il-admin', 'drewsIL', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('drews_il_nonce'),
            'strings' => array(
                'confirm_link' => 'Are you sure you want to convert all instances of "%s" to link to "%s" across all %s?',
                'confirm_remove' => 'Are you sure you want to remove this internal linking rule?',
                'confirm_re_run' => 'Are you sure you want to re-apply this linking rule?',
                'processing' => 'Processing...',
                'success' => 'Success!',
                'error' => 'Error occurred'
            )
        ));
    }
    
    public function admin_page() {
        global $wpdb;
        
        // Try to get cached links first
        $cache_key = 'drews_il_links_list';
        $links = wp_cache_get($cache_key);
        
        if (false === $links) {
            // Note: Direct database call is necessary for custom table operations
            // WordPress doesn't provide built-in caching for custom tables
            $links = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}drews_internal_links` ORDER BY created_at DESC");
            // Cache for 5 minutes
            wp_cache_set($cache_key, $links, '', 300);
        }
        
        include DREWS_IL_PLUGIN_PATH . 'templates/admin-page.php';
    }
    
    public function ajax_create_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['anchor_text']) || !isset($_POST['target_url']) || !isset($_POST['scope'])) {
            wp_send_json_error('Missing required fields');
        }
        
        $anchor_text = sanitize_text_field(wp_unslash($_POST['anchor_text']));
        $target_url = esc_url_raw(wp_unslash($_POST['target_url']));
        $scope = sanitize_text_field(wp_unslash($_POST['scope']));
        
        if (empty($anchor_text) || empty($target_url)) {
            wp_send_json_error('Missing required fields');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        // Check if this anchor text already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM `{$wpdb->prefix}drews_internal_links` WHERE anchor_text = %s",
            $anchor_text
        ));
        
        if ($existing) {
            wp_send_json_error('This anchor text already exists');
        }
        
        // Insert the new rule
        $result = $wpdb->insert(
            $table_name,
            array(
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'scope' => $scope
            ),
            array('%s', '%s', '%s')
        );
        
        if ($result === false) {
            wp_send_json_error('Database error');
        }
        
        $link_id = $wpdb->insert_id;
        
        // Start the linking process
        $this->process_linking($anchor_text, $target_url, $scope, $link_id);
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success(array(
            'message' => 'Internal linking rule created successfully',
            'link_id' => $link_id
        ));
    }
    
    public function ajax_remove_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        // Get the link details before removing
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}drews_internal_links` WHERE id = %d",
            $link_id
        ));
        
        if (!$link) {
            wp_send_json_error('Link not found');
        }
        
        // Remove the links from content first
        $this->remove_links_from_content($link->anchor_text, $link->target_url, $link->scope);
        
        // Remove the rule from database
        $result = $wpdb->delete(
            $table_name,
            array('id' => $link_id),
            array('%d')
        );
        
        if ($result === false) {
            wp_send_json_error('Database error');
        }
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success('Internal linking rule removed successfully');
    }
    
    public function ajax_re_run_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}drews_internal_links` WHERE id = %d",
            $link_id
        ));
        
        if (!$link) {
            wp_send_json_error('Link not found');
        }
        
        // Re-apply the linking
        $this->process_linking($link->anchor_text, $link->target_url, $link->scope, $link_id);
        
        // Update last_run timestamp
        $wpdb->update(
            $table_name,
            array('last_run' => current_time('mysql')),
            array('id' => $link_id),
            array('%s'),
            array('%d')
        );
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success('Linking rule re-applied successfully');
    }
    
    public function ajax_get_progress() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        $progress_key = 'drews_il_progress_' . $link_id;
        $progress = get_transient($progress_key);
        
        if ($progress === false) {
            $progress = array('status' => 'completed');
        }
        
        wp_send_json_success($progress);
    }
    
    private function process_linking($anchor_text, $target_url, $scope, $link_id) {
        $progress_key = 'drews_il_progress_' . $link_id;
        
        // Set initial progress
        set_transient($progress_key, array(
            'status' => 'processing',
            'current' => 0,
            'total' => 0,
            'message' => 'Starting...'
        ), HOUR_IN_SECONDS);
        
        // Determine post types to process
        $post_types = array();
        if ($scope === 'posts' || $scope === 'both') {
            $post_types[] = 'post';
        }
        if ($scope === 'pages' || $scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => 'publish'
        ));
        
        $total = count($posts);
        $processed = 0;
        $updated = 0;
        
        set_transient($progress_key, array(
            'status' => 'processing',
            'current' => 0,
            'total' => $total,
            'message' => 'Processing posts...'
        ), HOUR_IN_SECONDS);
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $original_content = $content;
            
            // Create the link HTML
            $link_html = '<a href="' . esc_url($target_url) . '">' . esc_html($anchor_text) . '</a>';
            
            // Replace all instances (case insensitive)
            $pattern = '/\b' . preg_quote($anchor_text, '/') . '\b/i';
            $new_content = preg_replace($pattern, $link_html, $content);
            
            // Only update if content changed
            if ($new_content !== $content) {
                wp_update_post(array(
                    'ID' => $post->ID,
                    'post_content' => $new_content
                ));
                $updated++;
            }
            
            $processed++;
            
            // Update progress every 10 posts
            if ($processed % 10 === 0) {
                set_transient($progress_key, array(
                    'status' => 'processing',
                    'current' => $processed,
                    'total' => $total,
                    'message' => "Processed $processed of $total posts..."
                ), HOUR_IN_SECONDS);
            }
        }
        
        // Set final progress
        set_transient($progress_key, array(
            'status' => 'completed',
            'current' => $total,
            'total' => $total,
            'message' => "Completed! Updated $updated posts.",
            'updated' => $updated
        ), HOUR_IN_SECONDS);
    }
    
    private function remove_links_from_content($anchor_text, $target_url, $scope) {
        // Determine post types to process
        $post_types = array();
        if ($scope === 'posts' || $scope === 'both') {
            $post_types[] = 'post';
        }
        if ($scope === 'pages' || $scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => 'publish'
        ));
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $original_content = $content;
            
            // Remove links that match our pattern
            $pattern = '/<a[^>]*href=["\']' . preg_quote($target_url, '/') . '["\'][^>]*>' . preg_quote($anchor_text, '/') . '<\/a>/i';
            $new_content = preg_replace($pattern, $anchor_text, $content);
            
            // Only update if content changed
            if ($new_content !== $content) {
                wp_update_post(array(
                    'ID' => $post->ID,
                    'post_content' => $new_content
                ));
            }
        }
    }
}

// Initialize the plugin
new DrewsSimpleInternalLinking();
