jQuery(document).ready(function($) {
    'use strict';
    
    let currentLinkId = null;
    let progressInterval = null;
    
    // Form submission
    $('#drews-il-form').on('submit', function(e) {
        e.preventDefault();
        
        const anchorText = $('#anchor_text').val().trim();
        const targetUrl = $('#target_url').val().trim();
        const scope = $('#scope').val();
        
        if (!anchorText || !targetUrl) {
            showNotice('Please fill in all required fields.', 'error');
            return;
        }
        
        // Show confirmation modal
        showConfirmationModal(
            'Create Internal Link',
            drewsIL.strings.confirm_link.replace('%s', anchorText).replace('%s', targetUrl).replace('%s', getScopeLabel(scope)),
            function() {
                createInternalLink(anchorText, targetUrl, scope);
            }
        );
    });
    
    // Re-run button
    $(document).on('click', '.drews-il-re-run', function() {
        const linkId = $(this).data('link-id');
        const row = $(this).closest('tr');
        const anchorText = row.find('td:first strong').text();
        const targetUrl = row.find('td:nth-child(2) a').attr('href');
        const scope = getScopeFromLabel(row.find('td:nth-child(3)').text());
        
        showConfirmationModal(
            'Re-apply Linking Rule',
            drewsIL.strings.confirm_re_run,
            function() {
                reRunLinking(linkId, anchorText, targetUrl, scope);
            }
        );
    });
    
    // Remove button
    $(document).on('click', '.drews-il-remove', function() {
        const linkId = $(this).data('link-id');
        const row = $(this).closest('tr');
        const anchorText = row.find('td:first strong').text();
        
        showConfirmationModal(
            'Remove Internal Link',
            drewsIL.strings.confirm_remove,
            function() {
                removeInternalLink(linkId, row);
            }
        );
    });
    
    // Modal close events
    $('#drews-il-modal-cancel, #drews-il-modal').on('click', function(e) {
        if (e.target === this) {
            hideModal();
        }
    });
    
    // Helper functions
    function getScopeLabel(scope) {
        const labels = {
            'both': 'Pages and Posts',
            'pages': 'Pages Only',
            'posts': 'Posts Only'
        };
        return labels[scope] || scope;
    }
    
    function getScopeFromLabel(label) {
        if (label.includes('Pages & Posts')) return 'both';
        if (label.includes('Pages Only')) return 'pages';
        if (label.includes('Posts Only')) return 'posts';
        return 'both';
    }
    
    function showConfirmationModal(title, message, onConfirm) {
        $('#drews-il-modal-title').text(title);
        $('#drews-il-modal-message').text(message);
        $('#drews-il-modal').show();
        
        // Remove previous event handlers
        $('#drews-il-modal-confirm').off('click');
        
        // Add new event handler
        $('#drews-il-modal-confirm').on('click', function() {
            hideModal();
            onConfirm();
        });
    }
    
    function hideModal() {
        $('#drews-il-modal').hide();
    }
    
    function showNotice(message, type = 'success') {
        const noticeClass = type === 'error' ? 'drews-il-notice error' : 'drews-il-notice success';
        const notice = $('<div class="' + noticeClass + '">' + message + '</div>');
        
        $('.drews-il-container').prepend(notice);
        
        // Auto-remove after 5 seconds
        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    function showProgress() {
        $('#drews-il-progress').show();
        $('.drews-il-submit').addClass('loading');
        $('.drews-il-submit .button-text').text(drewsIL.strings.processing);
    }
    
    function hideProgress() {
        $('#drews-il-progress').hide();
        $('.drews-il-submit').removeClass('loading');
        $('.drews-il-submit .button-text').text('Create Internal Link');
    }
    
    function updateProgress(progress) {
        if (progress.status === 'processing') {
            const percentage = progress.total > 0 ? (progress.current / progress.total) * 100 : 0;
            $('.drews-il-progress-fill').css('width', percentage + '%');
            $('.drews-il-progress-text').text(progress.message);
        } else if (progress.status === 'completed') {
            $('.drews-il-progress-fill').css('width', '100%');
            $('.drews-il-progress-text').text(progress.message);
            
            setTimeout(function() {
                hideProgress();
                if (progress.updated > 0) {
                    showNotice(`Successfully updated ${progress.updated} posts/pages!`, 'success');
                }
            }, 2000);
        }
    }
    
    // AJAX Functions
    function createInternalLink(anchorText, targetUrl, scope) {
        showProgress();
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_create_link',
                nonce: drewsIL.nonce,
                anchor_text: anchorText,
                target_url: targetUrl,
                scope: scope
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message, 'success');
                    currentLinkId = response.data.link_id;
                    
                    // Start progress monitoring
                    startProgressMonitoring(currentLinkId);
                    
                    // Reset form
                    $('#drews-il-form')[0].reset();
                    
                    // Reload page after a delay to show new entry
                    setTimeout(function() {
                        location.reload();
                    }, 3000);
                } else {
                    hideProgress();
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                hideProgress();
                showNotice('An error occurred while creating the internal link.', 'error');
            }
        });
    }
    
    function reRunLinking(linkId, anchorText, targetUrl, scope) {
        currentLinkId = linkId;
        showProgress();
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_re_run_link',
                nonce: drewsIL.nonce,
                link_id: linkId
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data, 'success');
                    startProgressMonitoring(linkId);
                } else {
                    hideProgress();
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                hideProgress();
                showNotice('An error occurred while re-applying the linking rule.', 'error');
            }
        });
    }
    
    function removeInternalLink(linkId, row) {
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_remove_link',
                nonce: drewsIL.nonce,
                link_id: linkId
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data, 'success');
                    row.fadeOut(function() {
                        $(this).remove();
                        
                        // Check if table is empty
                        if ($('tbody tr').length === 0) {
                            location.reload();
                        }
                    });
                } else {
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                showNotice('An error occurred while removing the internal link.', 'error');
            }
        });
    }
    
    function startProgressMonitoring(linkId) {
        if (progressInterval) {
            clearInterval(progressInterval);
        }
        
        progressInterval = setInterval(function() {
                    $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_get_progress',
                nonce: drewsIL.nonce,
                    link_id: linkId
                },
                success: function(response) {
                    if (response.success) {
                        updateProgress(response.data);
                        
                        if (response.data.status === 'completed') {
                            clearInterval(progressInterval);
                            progressInterval = null;
                        }
                    }
                },
                error: function() {
                    clearInterval(progressInterval);
                    progressInterval = null;
                    hideProgress();
                    showNotice('Error checking progress.', 'error');
                }
            });
        }, 1000);
    }
    
    // Clean up on page unload
    $(window).on('beforeunload', function() {
        if (progressInterval) {
            clearInterval(progressInterval);
        }
    });
});
