<?php
/**
 * Emergency Bulk Find & Replace Tool
 * 
 * This tool helps clean up malformed links created by the internal linking plugin
 * Use this to fix broken HTML patterns across all posts
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DrewsIL_BulkFindReplace {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_emergency_menu'));
        add_action('wp_ajax_drews_il_bulk_find_replace', array($this, 'ajax_bulk_find_replace'));
        add_action('wp_ajax_drews_il_preview_changes', array($this, 'ajax_preview_changes'));
    }
    
    public function add_emergency_menu() {
        // Add as a submenu under the main plugin page instead of themes
        add_submenu_page(
            'drews-simple-internal-linking',
            'Bulk Find & Replace',
            'Bulk Find & Replace',
            'manage_options',
            'drews-il-emergency',
            array($this, 'emergency_page')
        );
    }
    
    public function emergency_page() {
        ?>
        <div class="wrap">
            <h1>🚨 Emergency Find & Replace Tool</h1>
            
            <div class="notice notice-warning">
                <p><strong>⚠️ WARNING:</strong> This tool will modify content across your entire site. Always backup your database before proceeding!</p>
            </div>
            
            <div class="drews-il-emergency-container">
                <div class="drews-il-emergency-section">
                    <h2>Common Malformed Link Patterns</h2>
                    <p>Based on the reported issue, here are the most common patterns that need fixing:</p>
                    
                    <div class="pattern-examples">
                        <h3>Pattern 1: Broken anchor tag closures</h3>
                        <div class="pattern-box">
                            <strong>Find:</strong> <code>">text</code><br>
                            <strong>Replace with:</strong> <code>text</code><br>
                            <strong>Example:</strong> <code>">seo-impact.</code> → <code>seo-impact.</code>
                        </div>
                        
                        <h3>Pattern 2: Malformed URLs in text</h3>
                        <div class="pattern-box">
                            <strong>Find:</strong> <code>">off-page seo-1/">seo</code><br>
                            <strong>Replace with:</strong> <code>seo</code><br>
                            <strong>Example:</strong> <code>">off-page seo-1/">seo techniques</code> → <code>seo techniques</code>
                        </div>
                        
                        <h3>Pattern 3: Orphaned link fragments</h3>
                        <div class="pattern-box">
                            <strong>Find:</strong> <code>example.com/some-page/</code><br>
                            <strong>Replace with:</strong> <code>relevant text</code><br>
                            <strong>Example:</strong> <code>example.com/some-page/">broken-text.</code> → <code>relevant text broken-text.</code>
                        </div>
                    </div>
                </div>
                
                <div class="drews-il-emergency-section">
                    <h2>Custom Find & Replace</h2>
                    <form id="drews-il-find-replace-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="find_text">Find Text</label>
                                </th>
                                <td>
                                    <textarea id="find_text" name="find_text" rows="3" cols="50" 
                                              placeholder="Enter the text pattern to find (supports regex)"></textarea>
                                    <p class="description">Enter the exact text or regex pattern to find. Use <code>.*?</code> for wildcards.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="replace_text">Replace With</label>
                                </th>
                                <td>
                                    <textarea id="replace_text" name="replace_text" rows="3" cols="50" 
                                              placeholder="Enter the replacement text"></textarea>
                                    <p class="description">Enter what to replace the found text with. Leave empty to remove the text.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="post_types">Post Types</label>
                                </th>
                                <td>
                                    <select id="post_types" name="post_types" multiple>
                                        <option value="post" selected>Posts</option>
                                        <option value="page" selected>Pages</option>
                                    </select>
                                    <p class="description">Select which post types to search in.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="use_regex">Use Regex</label>
                                </th>
                                <td>
                                    <input type="checkbox" id="use_regex" name="use_regex" value="1">
                                    <label for="use_regex">Enable regex pattern matching</label>
                                    <p class="description">Check this if your "Find Text" contains regex patterns.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="divi_safe">Divi Safe Mode</label>
                                </th>
                                <td>
                                    <input type="checkbox" id="divi_safe" name="divi_safe" value="1" checked>
                                    <label for="divi_safe">Enable Divi compatibility mode</label>
                                    <p class="description"><strong>Recommended for Divi users:</strong> Skips posts with heavy Divi shortcode usage and validates changes to prevent breaking Divi's structure.</p>
                                </td>
                            </tr>
                        </table>
                        
                        <div class="drews-il-emergency-actions">
                            <button type="button" id="preview-changes" class="button button-secondary">
                                <span class="dashicons dashicons-visibility"></span>
                                Preview Changes
                            </button>
                            <button type="submit" id="execute-replace" class="button button-primary">
                                <span class="dashicons dashicons-update"></span>
                                Execute Find & Replace
                            </button>
                        </div>
                    </form>
                </div>
                
                <div id="preview-results" class="drews-il-emergency-section" style="display: none;">
                    <h2>Preview Results</h2>
                    <div id="preview-content"></div>
                </div>
                
                <div id="progress-section" class="drews-il-emergency-section" style="display: none;">
                    <h2>Progress</h2>
                    <div class="drews-il-progress">
                        <div class="drews-il-progress-bar">
                            <div class="drews-il-progress-fill"></div>
                        </div>
                        <div class="drews-il-progress-text">Processing...</div>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .drews-il-emergency-container {
            max-width: 1200px;
        }
        
        .drews-il-emergency-section {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }
        
        .pattern-examples h3 {
            color: #d63638;
            margin-top: 20px;
        }
        
        .pattern-box {
            background: #f6f7f7;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            margin: 10px 0;
            font-family: monospace;
        }
        
        .pattern-box code {
            background: #fff;
            padding: 2px 4px;
            border-radius: 2px;
            border: 1px solid #ddd;
        }
        
        .drews-il-emergency-actions {
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid #eee;
        }
        
        .drews-il-emergency-actions .button {
            margin-right: 10px;
        }
        
        .drews-il-progress {
            margin-top: 20px;
        }
        
        .drews-il-progress-bar {
            width: 100%;
            height: 20px;
            background-color: #f1f1f1;
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 10px;
        }
        
        .drews-il-progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #d63638, #ff6b6b);
            width: 0%;
            transition: width 0.3s ease;
            border-radius: 10px;
        }
        
        .drews-il-progress-text {
            text-align: center;
            font-weight: 600;
            color: #666;
        }
        
        .preview-item {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            margin: 10px 0;
        }
        
        .preview-item h4 {
            margin-top: 0;
            color: #d63638;
        }
        
        .preview-before, .preview-after {
            margin: 10px 0;
        }
        
        .preview-before strong, .preview-after strong {
            display: block;
            margin-bottom: 5px;
        }
        
        .preview-content {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 2px;
            padding: 10px;
            font-family: monospace;
            white-space: pre-wrap;
            max-height: 200px;
            overflow-y: auto;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#preview-changes').on('click', function() {
                previewChanges();
            });
            
            $('#drews-il-find-replace-form').on('submit', function(e) {
                e.preventDefault();
                executeFindReplace();
            });
            
            function previewChanges() {
                const findText = $('#find_text').val();
                const replaceText = $('#replace_text').val();
                const postTypes = $('#post_types').val();
                const useRegex = $('#use_regex').is(':checked');
                const diviSafe = $('#divi_safe').is(':checked');
                
                if (!findText) {
                    alert('Please enter text to find.');
                    return;
                }
                
                $('#preview-results').show();
                $('#preview-content').html('<p>Searching for matches...</p>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'drews_il_preview_changes',
                        nonce: '<?php echo wp_create_nonce('drews_il_emergency_nonce'); ?>',
                        find_text: findText,
                        replace_text: replaceText,
                        post_types: postTypes,
                        use_regex: useRegex,
                        divi_safe: diviSafe
                    },
                    success: function(response) {
                        if (response.success) {
                            displayPreviewResults(response.data);
                        } else {
                            $('#preview-content').html('<p style="color: red;">Error: ' + response.data + '</p>');
                        }
                    },
                    error: function() {
                        $('#preview-content').html('<p style="color: red;">An error occurred while previewing changes.</p>');
                    }
                });
            }
            
            function displayPreviewResults(data) {
                let html = '<h3>Found ' + data.total_matches + ' matches in ' + data.affected_posts + ' posts</h3>';
                
                if (data.skipped_posts > 0) {
                    html += '<p style="color: orange;"><strong>⚠️ Skipped ' + data.skipped_posts + ' posts</strong> (Divi-heavy content or unsafe changes)</p>';
                }
                
                if (data.preview_items && data.preview_items.length > 0) {
                    data.preview_items.forEach(function(item) {
                        html += '<div class="preview-item">';
                        html += '<h4>' + item.post_title + ' (ID: ' + item.post_id + ')';
                        if (item.is_divi) {
                            html += ' <span style="color: #0073aa; font-size: 12px;">[Divi Content]</span>';
                        }
                        html += '</h4>';
                        html += '<div class="preview-before"><strong>Before:</strong><div class="preview-content">' + escapeHtml(item.before) + '</div></div>';
                        html += '<div class="preview-after"><strong>After:</strong><div class="preview-content">' + escapeHtml(item.after) + '</div></div>';
                        html += '</div>';
                    });
                }
                
                $('#preview-content').html(html);
            }
            
            function executeFindReplace() {
                const findText = $('#find_text').val();
                const replaceText = $('#replace_text').val();
                const postTypes = $('#post_types').val();
                const useRegex = $('#use_regex').is(':checked');
                const diviSafe = $('#divi_safe').is(':checked');
                
                if (!findText) {
                    alert('Please enter text to find.');
                    return;
                }
                
                if (!confirm('Are you sure you want to execute this find and replace operation? This will modify content across your site.')) {
                    return;
                }
                
                $('#progress-section').show();
                $('#execute-replace').prop('disabled', true);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'drews_il_bulk_find_replace',
                        nonce: '<?php echo wp_create_nonce('drews_il_emergency_nonce'); ?>',
                        find_text: findText,
                        replace_text: replaceText,
                        post_types: postTypes,
                        use_regex: useRegex,
                        divi_safe: diviSafe
                    },
                    success: function(response) {
                        if (response.success) {
                            let message = 'Success! Updated ' + response.data.updated_posts + ' posts with ' + response.data.total_replacements + ' replacements.';
                            if (response.data.skipped_posts > 0) {
                                message += '\n\nSkipped ' + response.data.skipped_posts + ' posts (Divi-heavy content or unsafe changes).';
                            }
                            alert(message);
                            location.reload();
                        } else {
                            alert('Error: ' + response.data);
                        }
                    },
                    error: function() {
                        alert('An error occurred while executing the find and replace operation.');
                    },
                    complete: function() {
                        $('#execute-replace').prop('disabled', false);
                        $('#progress-section').hide();
                    }
                });
            }
            
            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }
        });
        </script>
        <?php
    }
    
    public function ajax_preview_changes() {
        check_ajax_referer('drews_il_emergency_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $find_text = sanitize_textarea_field(wp_unslash($_POST['find_text']));
        $replace_text = sanitize_textarea_field(wp_unslash($_POST['replace_text']));
        $post_types = array_map('sanitize_text_field', wp_unslash($_POST['post_types']));
        $use_regex = isset($_POST['use_regex']) && $_POST['use_regex'] === '1';
        $divi_safe = isset($_POST['divi_safe']) && $_POST['divi_safe'] === '1';
        
        if (empty($find_text)) {
            wp_send_json_error('Find text is required');
        }
        
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => 'publish'
        ));
        
        $total_matches = 0;
        $affected_posts = 0;
        $skipped_posts = 0;
        $preview_items = array();
        
        foreach ($posts as $post) {
            $original_content = $post->post_content;
            
            // Skip Divi-heavy content if Divi-safe mode is enabled
            if ($divi_safe && $this->is_divi_heavy_content($original_content)) {
                $skipped_posts++;
                continue;
            }
            
            if ($use_regex) {
                $new_content = preg_replace('/' . $find_text . '/', $replace_text, $original_content);
                // CRITICAL: Check if preg_replace failed
                if ($new_content === null || $new_content === false) {
                    $skipped_posts++;
                    continue; // Skip this post if regex failed
                }
            } else {
                $new_content = str_replace($find_text, $replace_text, $original_content);
            }
            
            // CRITICAL: Never allow empty content
            if (empty($new_content) || strlen($new_content) < 10) {
                $skipped_posts++;
                continue; // Skip if content would be deleted
            }
            
            // Validate the result if Divi-safe mode is enabled
            if ($divi_safe && !$this->is_safe_content($new_content, $original_content)) {
                $skipped_posts++;
                continue;
            }
            
            if ($new_content !== $original_content) {
                $affected_posts++;
                $matches = $use_regex ? preg_match_all('/' . $find_text . '/', $original_content) : substr_count($original_content, $find_text);
                $total_matches += $matches;
                
                if (count($preview_items) < 5) { // Show first 5 matches
                    $preview_items[] = array(
                        'post_id' => $post->ID,
                        'post_title' => $post->post_title,
                        'before' => $original_content,
                        'after' => $new_content,
                        'is_divi' => $this->is_divi_heavy_content($original_content)
                    );
                }
            }
        }
        
        wp_send_json_success(array(
            'total_matches' => $total_matches,
            'affected_posts' => $affected_posts,
            'skipped_posts' => $skipped_posts,
            'preview_items' => $preview_items
        ));
    }
    
    public function ajax_bulk_find_replace() {
        check_ajax_referer('drews_il_emergency_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $find_text = sanitize_textarea_field(wp_unslash($_POST['find_text']));
        $replace_text = sanitize_textarea_field(wp_unslash($_POST['replace_text']));
        $post_types = array_map('sanitize_text_field', wp_unslash($_POST['post_types']));
        $use_regex = isset($_POST['use_regex']) && $_POST['use_regex'] === '1';
        $divi_safe = isset($_POST['divi_safe']) && $_POST['divi_safe'] === '1';
        
        if (empty($find_text)) {
            wp_send_json_error('Find text is required');
        }
        
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => 'publish'
        ));
        
        $updated_posts = 0;
        $total_replacements = 0;
        $skipped_posts = 0;
        
        foreach ($posts as $post) {
            $original_content = $post->post_content;
            
            // Skip Divi-heavy content if Divi-safe mode is enabled
            if ($divi_safe && $this->is_divi_heavy_content($original_content)) {
                $skipped_posts++;
                continue;
            }
            
            if ($use_regex) {
                $new_content = preg_replace('/' . $find_text . '/', $replace_text, $original_content);
                // CRITICAL: Check if preg_replace failed
                if ($new_content === null || $new_content === false) {
                    $skipped_posts++;
                    continue; // Skip this post if regex failed
                }
            } else {
                $new_content = str_replace($find_text, $replace_text, $original_content);
            }
            
            // CRITICAL: Never allow empty content
            if (empty($new_content) || strlen($new_content) < 10) {
                $skipped_posts++;
                continue; // Skip if content would be deleted
            }
            
            // Validate the result if Divi-safe mode is enabled
            if ($divi_safe && !$this->is_safe_content($new_content, $original_content)) {
                $skipped_posts++;
                continue;
            }
            
            if ($new_content !== $original_content) {
                $result = wp_update_post(array(
                    'ID' => $post->ID,
                    'post_content' => $new_content
                ));
                
                if ($result && !is_wp_error($result)) {
                    $updated_posts++;
                    $matches = $use_regex ? preg_match_all('/' . $find_text . '/', $original_content) : substr_count($original_content, $find_text);
                    $total_replacements += $matches;
                }
            }
        }
        
        wp_send_json_success(array(
            'updated_posts' => $updated_posts,
            'total_replacements' => $total_replacements,
            'skipped_posts' => $skipped_posts
        ));
    }
    
    private function is_divi_heavy_content($content) {
        // Check if content is primarily Divi shortcodes
        $divi_shortcodes = array(
            '[et_pb_section',
            '[et_pb_row',
            '[et_pb_column',
            '[et_pb_text',
            '[et_pb_image',
            '[et_pb_button'
        );
        
        $shortcode_count = 0;
        foreach ($divi_shortcodes as $shortcode) {
            $shortcode_count += substr_count($content, $shortcode);
        }
        
        // If more than 50% of the content is Divi shortcodes, consider it Divi-heavy
        $total_length = strlen($content);
        return $shortcode_count > 0 && ($shortcode_count * 20) > ($total_length * 0.5);
    }
    
    private function is_safe_content($new_content, $original_content) {
        // Check for potential issues that could break Divi
        $issues = array();
        
        // Check for broken HTML tags
        if (substr_count($new_content, '<') !== substr_count($new_content, '>')) {
            $issues[] = 'Unmatched HTML tags';
        }
        
        // Check for broken shortcodes
        if (preg_match('/\[[^\]]*$/', $new_content)) {
            $issues[] = 'Broken shortcode';
        }
        
        // Check for excessive nesting (could break Divi)
        $nesting_depth = $this->get_max_nesting_depth($new_content);
        if ($nesting_depth > 10) {
            $issues[] = 'Excessive HTML nesting';
        }
        
        // Check if Divi shortcodes are still intact
        $divi_shortcodes = array('[et_pb_section', '[et_pb_row', '[et_pb_column');
        foreach ($divi_shortcodes as $shortcode) {
            if (strpos($original_content, $shortcode) !== false && strpos($new_content, $shortcode) === false) {
                $issues[] = 'Divi shortcode removed';
            }
        }
        
        // If there are issues, it's not safe
        return empty($issues);
    }
    
    private function get_max_nesting_depth($content) {
        $max_depth = 0;
        $current_depth = 0;
        $in_tag = false;
        
        for ($i = 0; $i < strlen($content); $i++) {
            $char = $content[$i];
            
            if ($char === '<' && $content[$i + 1] !== '/') {
                $in_tag = true;
                $current_depth++;
                $max_depth = max($max_depth, $current_depth);
            } elseif ($char === '>' && $in_tag) {
                $in_tag = false;
            } elseif ($char === '<' && $content[$i + 1] === '/') {
                $current_depth = max(0, $current_depth - 1);
            }
        }
        
        return $max_depth;
    }
}

// Initialize the emergency tool
new DrewsIL_BulkFindReplace();
