<?php
/**
 * Debug Content Tool
 * 
 * This tool helps debug why the plugin isn't finding text in posts
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DrewsIL_DebugContent {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_debug_menu'));
        add_action('wp_ajax_drews_il_debug_content', array($this, 'ajax_debug_content'));
    }
    
    public function add_debug_menu() {
        add_submenu_page(
            'drews-simple-internal-linking',
            'Debug Content',
            'Debug Content',
            'manage_options',
            'drews-il-debug',
            array($this, 'debug_page')
        );
    }
    
    public function debug_page() {
        ?>
        <div class="wrap">
            <h1>🔍 Debug Content Tool</h1>
            
            <div class="notice notice-info">
                <p><strong>This tool helps debug why the plugin isn't finding text in posts.</strong></p>
            </div>
            
            <div class="drews-il-debug-container">
                <div class="drews-il-debug-section">
                    <h2>Search for Text in Posts</h2>
                    <form id="drews-il-debug-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="search_text">Text to Search For</label>
                                </th>
                                <td>
                                    <input type="text" id="search_text" name="search_text" class="regular-text" 
                                           placeholder="e.g., Southbury, 1995" value="Southbury">
                                    <p class="description">Enter the text you're trying to find in posts.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="post_types">Post Types</label>
                                </th>
                                <td>
                                    <select id="post_types" name="post_types" multiple>
                                        <option value="post" selected>Posts</option>
                                        <option value="page" selected>Pages</option>
                                    </select>
                                    <p class="description">Select which post types to search in.</p>
                                </td>
                            </tr>
                        </table>
                        
                        <div class="drews-il-debug-actions">
                            <button type="button" id="debug-search" class="button button-primary">
                                <span class="dashicons dashicons-search"></span>
                                Search Content
                            </button>
                        </div>
                    </form>
                </div>
                
                <div id="debug-results" class="drews-il-debug-section" style="display: none;">
                    <h2>Search Results</h2>
                    <div id="debug-content"></div>
                </div>
            </div>
        </div>
        
        <style>
        .drews-il-debug-container {
            max-width: 1200px;
        }
        
        .drews-il-debug-section {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }
        
        .debug-item {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            margin: 10px 0;
        }
        
        .debug-item h4 {
            margin-top: 0;
            color: #d63638;
        }
        
        .debug-meta {
            font-size: 12px;
            color: #666;
            margin: 5px 0;
        }
        
        .debug-content {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 2px;
            padding: 10px;
            font-family: monospace;
            white-space: pre-wrap;
            max-height: 300px;
            overflow-y: auto;
            margin: 10px 0;
        }
        
        .debug-highlight {
            background: yellow;
            font-weight: bold;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#debug-search').on('click', function() {
                const searchText = $('#search_text').val();
                const postTypes = $('#post_types').val();
                
                if (!searchText) {
                    alert('Please enter text to search for.');
                    return;
                }
                
                $('#debug-results').show();
                $('#debug-content').html('<p>Searching for "' + searchText + '"...</p>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'drews_il_debug_content',
                        nonce: '<?php echo wp_create_nonce('drews_il_debug_nonce'); ?>',
                        search_text: searchText,
                        post_types: postTypes
                    },
                    success: function(response) {
                        if (response.success) {
                            displayDebugResults(response.data);
                        } else {
                            $('#debug-content').html('<p style="color: red;">Error: ' + response.data + '</p>');
                        }
                    },
                    error: function() {
                        $('#debug-content').html('<p style="color: red;">An error occurred while searching.</p>');
                    }
                });
            });
            
            function displayDebugResults(data) {
                let html = '<h3>Found ' + data.total_matches + ' matches in ' + data.affected_posts + ' posts</h3>';
                
                if (data.debug_info) {
                    html += '<div class="debug-meta" style="background: #f0f0f1; padding: 10px; border-radius: 4px; margin: 10px 0;">';
                    html += '<strong>Debug Info:</strong><br>';
                    html += 'Total posts searched: ' + data.debug_info.total_posts_searched + '<br>';
                    html += 'Search text: "' + data.debug_info.search_text + '"<br>';
                    html += 'Post types: ' + data.debug_info.post_types.join(', ');
                    html += '</div>';
                }
                
                if (data.posts && data.posts.length > 0) {
                    data.posts.forEach(function(post) {
                        html += '<div class="debug-item">';
                        html += '<h4>' + post.post_title + ' (ID: ' + post.post_id + ')</h4>';
                        html += '<div class="debug-meta">';
                        html += 'Post Type: ' + post.post_type + '<br>';
                        html += 'Content Length: ' + post.content_length + ' characters<br>';
                        html += 'Matches: ' + post.matches + '<br>';
                        html += 'Status: ' + post.post_status;
                        html += '</div>';
                        html += '<div class="debug-content">' + escapeHtml(post.content_preview) + '</div>';
                        html += '</div>';
                    });
                } else {
                    html += '<p style="color: orange;">No matches found. This could mean:</p>';
                    html += '<ul>';
                    html += '<li>The text doesn\'t exist in the selected post types</li>';
                    html += '<li>The text is stored in a different format (HTML entities, etc.)</li>';
                    html += '<li>The text is in a different field (title, excerpt, etc.)</li>';
                    html += '<li>There\'s an issue with the search logic</li>';
                    html += '</ul>';
                }
                
                $('#debug-content').html(html);
            }
            
            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }
        });
        </script>
        <?php
    }
    
    public function ajax_debug_content() {
        check_ajax_referer('drews_il_debug_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $search_text = sanitize_text_field(wp_unslash($_POST['search_text']));
        $post_types = array_map('sanitize_text_field', wp_unslash($_POST['post_types']));
        
        if (empty($search_text)) {
            wp_send_json_error('Search text is required');
        }
        
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => array('publish', 'draft', 'private', 'pending')
        ));
        
        $total_matches = 0;
        $affected_posts = 0;
        $debug_posts = array();
        
        // Debug: Show total posts being searched
        $debug_info = array(
            'total_posts_searched' => count($posts),
            'search_text' => $search_text,
            'post_types' => $post_types
        );
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $matches = substr_count(strtolower($content), strtolower($search_text));
            
            if ($matches > 0) {
                $affected_posts++;
                $total_matches += $matches;
                
                // Create a preview of the content with the search text highlighted
                $content_preview = $content;
                $content_preview = str_ireplace($search_text, '<span class="debug-highlight">' . $search_text . '</span>', $content_preview);
                
                // Truncate if too long
                if (strlen($content_preview) > 1000) {
                    $content_preview = substr($content_preview, 0, 1000) . '...';
                }
                
                $debug_posts[] = array(
                    'post_id' => $post->ID,
                    'post_title' => $post->post_title,
                    'post_type' => $post->post_type,
                    'post_status' => $post->post_status,
                    'content_length' => strlen($content),
                    'matches' => $matches,
                    'content_preview' => $content_preview
                );
            }
        }
        
        wp_send_json_success(array(
            'total_matches' => $total_matches,
            'affected_posts' => $affected_posts,
            'posts' => $debug_posts,
            'debug_info' => $debug_info
        ));
    }
}

// Initialize the debug tool
new DrewsIL_DebugContent();
