<?php
/**
 * Plugin Name: Drew's Simple Internal Linking
 * Plugin URI: https://chapin.io/drews-simple-internal-linking-wordpress-plugin/
 * Description: A simple plugin to automatically create internal links throughout your WordPress site. <a href="admin.php?page=drews-simple-internal-linking">Go to Dashboard</a>
 * Version: 1.0.3
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Author: Drew Chapin
 * Author URI: https://chapin.io
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: drews-simple-internal-linking
 * 
 * @package DrewsSimpleInternalLinking
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('DREWS_IL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DREWS_IL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('DREWS_IL_VERSION', '1.0.3');

class DrewsSimpleInternalLinking {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_drews_il_create_link', array($this, 'ajax_create_link'));
        add_action('wp_ajax_drews_il_remove_link', array($this, 'ajax_remove_link'));
        add_action('wp_ajax_drews_il_re_run_link', array($this, 'ajax_re_run_link'));
        add_action('wp_ajax_drews_il_get_progress', array($this, 'ajax_get_progress'));
        add_action('wp_ajax_drews_il_bulk_revert', array($this, 'ajax_bulk_revert'));
        add_action('wp_ajax_drews_il_backup_content', array($this, 'ajax_backup_content'));
        
        // Create database table on activation
        register_activation_hook(__FILE__, array($this, 'create_database_table'));
    }
    
    public function create_database_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'drews_internal_links';
        $backup_table_name = $wpdb->prefix . 'drews_internal_links_backup';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            anchor_text varchar(255) NOT NULL,
            target_url varchar(500) NOT NULL,
            scope varchar(20) NOT NULL DEFAULT 'both',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            last_run datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        $backup_sql = "CREATE TABLE $backup_table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            post_content longtext NOT NULL,
            backup_date datetime DEFAULT CURRENT_TIMESTAMP,
            link_rule_id mediumint(9) NOT NULL,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY link_rule_id (link_rule_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        dbDelta($backup_sql);
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'themes.php',
            'Drew\'s Simple Internal Linking',
            'Internal Linking',
            'manage_options',
            'drews-simple-internal-linking',
            array($this, 'admin_page')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'appearance_page_drews-simple-internal-linking') {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('drews-il-admin', DREWS_IL_PLUGIN_URL . 'js/admin.js', array('jquery'), DREWS_IL_VERSION, true);
        wp_enqueue_style('drews-il-admin', DREWS_IL_PLUGIN_URL . 'css/admin.css', array(), DREWS_IL_VERSION);
        
        wp_localize_script('drews-il-admin', 'drewsIL', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('drews_il_nonce'),
            'strings' => array(
                'confirm_link' => 'Are you sure you want to convert all instances of "%s" to link to "%s" across all %s?',
                'confirm_remove' => 'Are you sure you want to remove this internal linking rule?',
                'confirm_re_run' => 'Are you sure you want to re-apply this linking rule?',
                'processing' => 'Processing...',
                'success' => 'Success!',
                'error' => 'Error occurred'
            )
        ));
    }
    
    public function admin_page() {
        global $wpdb;
        
        // Try to get cached links first
        $cache_key = 'drews_il_links_list';
        $links = wp_cache_get($cache_key);
        
        if (false === $links) {
            // Note: Direct database call is necessary for custom table operations
            // WordPress doesn't provide built-in caching for custom tables
            $links = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}drews_internal_links` ORDER BY created_at DESC");
            // Cache for 5 minutes
            wp_cache_set($cache_key, $links, '', 300);
        }
        
        include DREWS_IL_PLUGIN_PATH . 'templates/admin-page.php';
    }
    
    public function ajax_create_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['anchor_text']) || !isset($_POST['target_url']) || !isset($_POST['scope'])) {
            wp_send_json_error('Missing required fields');
        }
        
        $anchor_text = sanitize_text_field(wp_unslash($_POST['anchor_text']));
        $target_url = esc_url_raw(wp_unslash($_POST['target_url']));
        $scope = sanitize_text_field(wp_unslash($_POST['scope']));
        
        if (empty($anchor_text) || empty($target_url)) {
            wp_send_json_error('Missing required fields');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        // Check if this anchor text already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM `{$wpdb->prefix}drews_internal_links` WHERE anchor_text = %s",
            $anchor_text
        ));
        
        if ($existing) {
            wp_send_json_error('This anchor text already exists');
        }
        
        // Insert the new rule
        $result = $wpdb->insert(
            $table_name,
            array(
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'scope' => $scope
            ),
            array('%s', '%s', '%s')
        );
        
        if ($result === false) {
            wp_send_json_error('Database error');
        }
        
        $link_id = $wpdb->insert_id;
        
        // Start the linking process
        $this->process_linking($anchor_text, $target_url, $scope, $link_id);
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success(array(
            'message' => 'Internal linking rule created successfully',
            'link_id' => $link_id
        ));
    }
    
    public function ajax_remove_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        // Get the link details before removing
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}drews_internal_links` WHERE id = %d",
            $link_id
        ));
        
        if (!$link) {
            wp_send_json_error('Link not found');
        }
        
        // Remove the links from content first
        $this->remove_links_from_content($link->anchor_text, $link->target_url, $link->scope);
        
        // Remove the rule from database
        $result = $wpdb->delete(
            $table_name,
            array('id' => $link_id),
            array('%d')
        );
        
        if ($result === false) {
            wp_send_json_error('Database error');
        }
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success('Internal linking rule removed successfully');
    }
    
    public function ajax_re_run_link() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'drews_internal_links';
        
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}drews_internal_links` WHERE id = %d",
            $link_id
        ));
        
        if (!$link) {
            wp_send_json_error('Link not found');
        }
        
        // Create backups before re-running
        $this->create_backups($link->anchor_text, $link->scope, $link_id);
        
        // Re-apply the linking
        $this->process_linking($link->anchor_text, $link->target_url, $link->scope, $link_id);
        
        // Update last_run timestamp
        $wpdb->update(
            $table_name,
            array('last_run' => current_time('mysql')),
            array('id' => $link_id),
            array('%s'),
            array('%d')
        );
        
        // Clear the links cache
        wp_cache_delete('drews_il_links_list');
        
        wp_send_json_success('Linking rule re-applied successfully');
    }
    
    public function ajax_get_progress() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        $progress_key = 'drews_il_progress_' . $link_id;
        $progress = get_transient($progress_key);
        
        if ($progress === false) {
            $progress = array('status' => 'completed');
        }
        
        wp_send_json_success($progress);
    }
    
    public function ajax_bulk_revert() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $backup_table = $wpdb->prefix . 'drews_internal_links_backup';
        
        // Get all backups for this link rule
        $backups = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $backup_table WHERE link_rule_id = %d ORDER BY backup_date DESC",
            $link_id
        ));
        
        if (empty($backups)) {
            wp_send_json_error('No backups found for this link rule');
        }
        
        $reverted = 0;
        $errors = 0;
        
        foreach ($backups as $backup) {
            $result = wp_update_post(array(
                'ID' => $backup->post_id,
                'post_content' => $backup->post_content
            ));
            
            if ($result && !is_wp_error($result)) {
                $reverted++;
            } else {
                $errors++;
            }
        }
        
        // Remove the backups after successful revert
        if ($errors === 0) {
            $wpdb->delete($backup_table, array('link_rule_id' => $link_id), array('%d'));
        }
        
        wp_send_json_success(array(
            'message' => "Reverted $reverted posts successfully" . ($errors > 0 ? " ($errors errors occurred)" : ""),
            'reverted' => $reverted,
            'errors' => $errors
        ));
    }
    
    public function ajax_backup_content() {
        check_ajax_referer('drews_il_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_POST['link_id'])) {
            wp_send_json_error('Missing link ID');
        }
        $link_id = intval(wp_unslash($_POST['link_id']));
        
        global $wpdb;
        $backup_table = $wpdb->prefix . 'drews_internal_links_backup';
        
        // Get the link rule
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM `{$wpdb->prefix}drews_internal_links` WHERE id = %d",
            $link_id
        ));
        
        if (!$link) {
            wp_send_json_error('Link rule not found');
        }
        
        // Determine post types to backup
        $post_types = array();
        if ($link->scope === 'posts' || $link->scope === 'both') {
            $post_types[] = 'post';
        }
        if ($link->scope === 'pages' || $link->scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages that might be affected
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => array('publish', 'draft', 'private', 'pending')
        ));
        
        $backed_up = 0;
        
        foreach ($posts as $post) {
            // Check if this post contains the anchor text
            if (stripos($post->post_content, $link->anchor_text) !== false) {
                // Create backup
                $wpdb->insert(
                    $backup_table,
                    array(
                        'post_id' => $post->ID,
                        'post_content' => $post->post_content,
                        'link_rule_id' => $link_id
                    ),
                    array('%d', '%s', '%d')
                );
                $backed_up++;
            }
        }
        
        wp_send_json_success(array(
            'message' => "Backed up $backed_up posts before applying changes",
            'backed_up' => $backed_up
        ));
    }
    
    private function create_backups($anchor_text, $scope, $link_id) {
        global $wpdb;
        $backup_table = $wpdb->prefix . 'drews_internal_links_backup';
        
        // Determine post types to backup
        $post_types = array();
        if ($scope === 'posts' || $scope === 'both') {
            $post_types[] = 'post';
        }
        if ($scope === 'pages' || $scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages that might be affected
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => array('publish', 'draft', 'private', 'pending')
        ));
        
        foreach ($posts as $post) {
            // Check if this post contains the anchor text
            if (stripos($post->post_content, $anchor_text) !== false) {
                // Check if backup already exists for this post and link rule
                $existing = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM $backup_table WHERE post_id = %d AND link_rule_id = %d",
                    $post->ID, $link_id
                ));
                
                if (!$existing) {
                    // Create backup
                    $wpdb->insert(
                        $backup_table,
                        array(
                            'post_id' => $post->ID,
                            'post_content' => $post->post_content,
                            'link_rule_id' => $link_id
                        ),
                        array('%d', '%s', '%d')
                    );
                }
            }
        }
    }
    
    private function process_linking($anchor_text, $target_url, $scope, $link_id) {
        $progress_key = 'drews_il_progress_' . $link_id;
        
        // Set initial progress
        set_transient($progress_key, array(
            'status' => 'processing',
            'current' => 0,
            'total' => 0,
            'message' => 'Creating backups...'
        ), HOUR_IN_SECONDS);
        
        // Create backups before making changes
        $this->create_backups($anchor_text, $scope, $link_id);
        
        // Determine post types to process
        $post_types = array();
        if ($scope === 'posts' || $scope === 'both') {
            $post_types[] = 'post';
        }
        if ($scope === 'pages' || $scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => array('publish', 'draft', 'private', 'pending')
        ));
        
        $total = count($posts);
        $processed = 0;
        $updated = 0;
        
        set_transient($progress_key, array(
            'status' => 'processing',
            'current' => 0,
            'total' => $total,
            'message' => 'Processing posts...'
        ), HOUR_IN_SECONDS);
        
        $debug_info = array();
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $original_content = $content;
            
            $post_debug = array(
                'post_id' => $post->ID,
                'post_title' => $post->post_title,
                'content_length' => strlen($content),
                'has_anchor_text' => false,
                'skipped_reason' => '',
                'updated' => false
            );
            
            // EMERGENCY CHECK: Skip if content is suspiciously short or empty
            // This prevents re-processing already damaged pages
            if (empty($content) || strlen($content) < 50) {
                $post_debug['skipped_reason'] = 'Content too short or empty (possibly damaged): ' . strlen($content) . ' chars';
                $debug_info[] = $post_debug;
                $processed++;
                error_log('Drew IL: Skipping post ' . $post->ID . ' - content only ' . strlen($content) . ' chars');
                continue;
            }
            
            // Additional check for damaged content patterns
            if (trim($content) === '' || $content === '<!-- wp:paragraph -->\n<!-- /wp:paragraph -->') {
                $post_debug['skipped_reason'] = 'Content appears to be damaged or placeholder';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            // Skip if it's Divi shortcode content that might be problematic
            if ($this->is_divi_shortcode_content($content)) {
                $post_debug['skipped_reason'] = 'Divi shortcode content';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            // Check if the anchor text exists in the content
            if (stripos($content, $anchor_text) === false) {
                $post_debug['skipped_reason'] = 'Anchor text not found';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            $post_debug['has_anchor_text'] = true;
            
            // Create the link HTML
            $link_html = '<a href="' . esc_url($target_url) . '">' . esc_html($anchor_text) . '</a>';
            
            // CRITICAL FIX: Use MUCH simpler approach that works with all content types
            $new_content = $content;
            
            // Check if text already linked
            $already_linked = preg_match('/<a[^>]*>' . preg_quote($anchor_text, '/') . '<\/a>/i', $content);
            
            if (!$already_linked) {
                // Use simple str_ireplace with validation
                // This is safer than complex regex for WordPress content
                $temp_content = $content;
                
                // Count occurrences first
                $count = substr_count(strtolower($temp_content), strtolower($anchor_text));
                
                if ($count > 0) {
                    // Do the replacement
                    $new_content = str_ireplace($anchor_text, $link_html, $temp_content);
                    
                    // Validate the replacement worked correctly
                    if ($new_content === $temp_content) {
                        $post_debug['skipped_reason'] = 'No changes made - text might be in attributes';
                        $debug_info[] = $post_debug;
                        $processed++;
                        continue;
                    }
                } else {
                    $post_debug['skipped_reason'] = 'Text not found in content';
                    $debug_info[] = $post_debug;
                    $processed++;
                    continue;
                }
            } else {
                $post_debug['skipped_reason'] = 'Text already linked';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            // CRITICAL: Multiple safety checks
            if ($new_content === null || $new_content === false || $new_content === '') {
                $post_debug['skipped_reason'] = 'Replacement resulted in null/false/empty content';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            if (strlen($new_content) < 10) {
                $post_debug['skipped_reason'] = 'Content too short after replacement: ' . strlen($new_content) . ' chars';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            // Check content hasn't been destroyed (should be similar length)
            $length_diff = abs(strlen($new_content) - strlen($content));
            $expected_diff = strlen($link_html) - strlen($anchor_text);
            if ($length_diff > ($expected_diff * $count * 2)) {
                $post_debug['skipped_reason'] = 'Content length changed too much - possible corruption';
                $debug_info[] = $post_debug;
                $processed++;
                continue;
            }
            
            // Only update if content changed and is valid
            if ($new_content !== $content && !empty($new_content)) {
                // Log what we're about to do for debugging
                error_log('Drew IL: About to update post ' . $post->ID . ' - Original length: ' . strlen($content) . ', New length: ' . strlen($new_content));
                
                // FINAL SAFETY CHECK: Ensure content wasn't destroyed
                if (strlen($new_content) < (strlen($original_content) * 0.5)) {
                    $post_debug['skipped_reason'] = 'Content reduced by more than 50% - preventing damage';
                    error_log('Drew IL: PREVENTED DAMAGE to post ' . $post->ID . ' - content would be reduced from ' . strlen($original_content) . ' to ' . strlen($new_content) . ' chars');
                    $debug_info[] = $post_debug;
                    $processed++;
                    continue;
                }
                
                // Basic safety check - make sure we didn't break anything major
                if ($this->is_safe_to_update($new_content, $original_content)) {
                    $result = wp_update_post(array(
                        'ID' => $post->ID,
                        'post_content' => $new_content
                    ));
                    
                    if ($result && !is_wp_error($result)) {
                        $updated++;
                        $post_debug['updated'] = true;
                        error_log('Drew IL: Successfully updated post ' . $post->ID);
                    } else {
                        $error_msg = is_wp_error($result) ? $result->get_error_message() : 'Unknown error';
                        $post_debug['skipped_reason'] = 'Update failed: ' . $error_msg;
                        error_log('Drew IL: Failed to update post ' . $post->ID . ' - ' . $error_msg);
                    }
                } else {
                    $post_debug['skipped_reason'] = 'Failed safety check';
                    error_log('Drew IL: Post ' . $post->ID . ' failed safety check');
                }
            } else {
                $post_debug['skipped_reason'] = 'No changes after replacement';
            }
            
            $debug_info[] = $post_debug;
            $processed++;
            
            // Update progress every 10 posts
            if ($processed % 10 === 0) {
                set_transient($progress_key, array(
                    'status' => 'processing',
                    'current' => $processed,
                    'total' => $total,
                    'message' => "Processed $processed of $total posts..."
                ), HOUR_IN_SECONDS);
            }
        }
        
        // Set final progress with debug info
        set_transient($progress_key, array(
            'status' => 'completed',
            'current' => $total,
            'total' => $total,
            'message' => "Completed! Updated $updated posts. Processed $processed posts total.",
            'updated' => $updated,
            'debug' => array(
                'total_posts' => $total,
                'processed' => $processed,
                'updated' => $updated,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'scope' => $scope
            )
        ), HOUR_IN_SECONDS);
    }
    
    private function remove_links_from_content($anchor_text, $target_url, $scope) {
        // Determine post types to process
        $post_types = array();
        if ($scope === 'posts' || $scope === 'both') {
            $post_types[] = 'post';
        }
        if ($scope === 'pages' || $scope === 'both') {
            $post_types[] = 'page';
        }
        
        // Get all posts/pages
        $posts = get_posts(array(
            'post_type' => $post_types,
            'numberposts' => -1,
            'post_status' => array('publish', 'draft', 'private', 'pending')
        ));
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $original_content = $content;
            
            // Remove links that match our pattern
            $pattern = '/<a[^>]*href=["\']' . preg_quote($target_url, '/') . '["\'][^>]*>' . preg_quote($anchor_text, '/') . '<\/a>/i';
            $new_content = preg_replace($pattern, $anchor_text, $content);
            
            // CRITICAL: Check if preg_replace failed
            if ($new_content === null || $new_content === false) {
                continue; // Skip this post if regex failed
            }
            
            // CRITICAL: Never update with empty content
            if (empty($new_content) || strlen($new_content) < 10) {
                continue; // Skip if content would be deleted
            }
            
            // Only update if content changed and is valid
            if ($new_content !== $content && !empty($new_content)) {
                wp_update_post(array(
                    'ID' => $post->ID,
                    'post_content' => $new_content
                ));
            }
        }
    }
    
    private function is_divi_shortcode_content($content) {
        // Check if content is primarily Divi shortcodes
        $divi_shortcodes = array(
            '[et_pb_section',
            '[et_pb_row',
            '[et_pb_column',
            '[et_pb_text',
            '[et_pb_image',
            '[et_pb_button'
        );
        
        $shortcode_count = 0;
        foreach ($divi_shortcodes as $shortcode) {
            $shortcode_count += substr_count($content, $shortcode);
        }
        
        // If more than 50% of the content is Divi shortcodes, skip it
        $total_length = strlen($content);
        return $shortcode_count > 0 && ($shortcode_count * 20) > ($total_length * 0.5);
    }
    
    private function is_safe_to_update($new_content, $original_content) {
        // Basic safety check - only check for major issues
        $issues = array();
        
        // Check for broken HTML tags
        if (substr_count($new_content, '<') !== substr_count($new_content, '>')) {
            $issues[] = 'Unmatched HTML tags';
        }
        
        // Check for broken shortcodes
        if (preg_match('/\[[^\]]*$/', $new_content)) {
            $issues[] = 'Broken shortcode';
        }
        
        // If there are major issues, don't update
        return empty($issues);
    }
    
    private function get_max_nesting_depth($content) {
        $max_depth = 0;
        $current_depth = 0;
        $in_tag = false;
        
        for ($i = 0; $i < strlen($content); $i++) {
            $char = $content[$i];
            
            if ($char === '<' && $content[$i + 1] !== '/') {
                $in_tag = true;
                $current_depth++;
                $max_depth = max($max_depth, $current_depth);
            } elseif ($char === '>' && $in_tag) {
                $in_tag = false;
            } elseif ($char === '<' && $content[$i + 1] === '/') {
                $current_depth = max(0, $current_depth - 1);
            }
        }
        
        return $max_depth;
    }
    
    private function is_wordpress_safe($new_content, $original_content) {
        // Check for WordPress-specific issues
        $issues = array();
        
        // Check for broken WordPress block comments
        if (preg_match('/<!-- wp:[^>]*$/', $new_content)) {
            $issues[] = 'Broken WordPress block comment';
        }
        
        // Check for broken shortcodes
        if (preg_match('/\[[^\]]*$/', $new_content)) {
            $issues[] = 'Broken shortcode';
        }
        
        // Check if we accidentally modified WordPress block structure
        $original_blocks = preg_match_all('/<!-- wp:[^>]*-->/', $original_content);
        $new_blocks = preg_match_all('/<!-- wp:[^>]*-->/', $new_content);
        if ($original_blocks !== $new_blocks) {
            $issues[] = 'WordPress block structure modified';
        }
        
        // Check for excessive HTML changes (might indicate corruption)
        $html_diff = strlen($new_content) - strlen($original_content);
        if (abs($html_diff) > 1000) { // More than 1000 characters difference
            $issues[] = 'Excessive content change detected';
        }
        
        return empty($issues);
    }
}

// Initialize the plugin
new DrewsSimpleInternalLinking();

// Include emergency tools
require_once DREWS_IL_PLUGIN_PATH . 'bulk-find-replace-tool.php';
require_once DREWS_IL_PLUGIN_PATH . 'emergency-restore.php';
require_once DREWS_IL_PLUGIN_PATH . 'debug-content.php';
