<?php
/**
 * Emergency Restore Tool
 * 
 * This tool helps restore pages that were damaged by the internal linking plugin
 * Run this in your WordPress admin to restore from WordPress revisions
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DrewsIL_EmergencyRestore {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_emergency_menu'));
        add_action('wp_ajax_drews_il_restore_page', array($this, 'ajax_restore_page'));
    }
    
    public function add_emergency_menu() {
        add_submenu_page(
            'drews-simple-internal-linking',
            'Emergency Restore',
            'Emergency Restore',
            'manage_options',
            'drews-il-restore',
            array($this, 'restore_page')
        );
    }
    
    public function restore_page() {
        ?>
        <div class="wrap">
            <h1>🚨 Emergency Restore Tool</h1>
            
            <div class="notice notice-warning">
                <p><strong>⚠️ URGENT:</strong> This tool helps restore pages damaged by the internal linking plugin using WordPress revisions.</p>
            </div>
            
            <div class="drews-il-emergency-container">
                <div class="drews-il-emergency-section">
                    <h2>Restore Damaged Pages</h2>
                    <p>If a page was damaged by the internal linking plugin, you can restore it from a previous revision:</p>
                    
                    <form id="drews-il-restore-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="page_id">Select Page to Restore</label>
                                </th>
                                <td>
                                    <select id="page_id" name="page_id">
                                        <option value="">Select a page...</option>
                                        <?php
                                        $pages = get_posts(array(
                                            'post_type' => array('post', 'page'),
                                            'numberposts' => -1,
                                            'post_status' => 'publish'
                                        ));
                                        
                                        foreach ($pages as $page) {
                                            echo '<option value="' . $page->ID . '">' . esc_html($page->post_title) . ' (ID: ' . $page->ID . ')</option>';
                                        }
                                        ?>
                                    </select>
                                    <p class="description">Select the page that was damaged by the internal linking plugin.</p>
                                </td>
                            </tr>
                        </table>
                        
                        <div class="drews-il-emergency-actions">
                            <button type="button" id="check-revisions" class="button button-secondary">
                                <span class="dashicons dashicons-backup"></span>
                                Check Available Revisions
                            </button>
                        </div>
                    </form>
                </div>
                
                <div id="revisions-section" class="drews-il-emergency-section" style="display: none;">
                    <h2>Available Revisions</h2>
                    <div id="revisions-content"></div>
                </div>
            </div>
        </div>
        
        <style>
        .drews-il-emergency-container {
            max-width: 1200px;
        }
        
        .drews-il-emergency-section {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }
        
        .revision-item {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            margin: 10px 0;
        }
        
        .revision-item h4 {
            margin-top: 0;
            color: #d63638;
        }
        
        .revision-meta {
            font-size: 12px;
            color: #666;
            margin: 5px 0;
        }
        
        .revision-actions {
            margin-top: 10px;
        }
        
        .revision-actions .button {
            margin-right: 10px;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#check-revisions').on('click', function() {
                const pageId = $('#page_id').val();
                
                if (!pageId) {
                    alert('Please select a page first.');
                    return;
                }
                
                $('#revisions-section').show();
                $('#revisions-content').html('<p>Loading revisions...</p>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'drews_il_restore_page',
                        nonce: '<?php echo wp_create_nonce('drews_il_restore_nonce'); ?>',
                        page_id: pageId
                    },
                    success: function(response) {
                        if (response.success) {
                            displayRevisions(response.data);
                        } else {
                            $('#revisions-content').html('<p style="color: red;">Error: ' + response.data + '</p>');
                        }
                    },
                    error: function() {
                        $('#revisions-content').html('<p style="color: red;">An error occurred while loading revisions.</p>');
                    }
                });
            });
            
            function displayRevisions(data) {
                let html = '<h3>Found ' + data.revisions.length + ' revisions for this page</h3>';
                
                if (data.revisions.length === 0) {
                    html += '<p style="color: orange;">No revisions found for this page. You may need to restore from a backup.</p>';
                } else {
                    data.revisions.forEach(function(revision) {
                        html += '<div class="revision-item">';
                        html += '<h4>Revision from ' + revision.date + '</h4>';
                        html += '<div class="revision-meta">';
                        html += 'Author: ' + revision.author + '<br>';
                        html += 'Status: ' + revision.status + '<br>';
                        html += 'Changes: ' + revision.changes + ' words';
                        html += '</div>';
                        html += '<div class="revision-actions">';
                        html += '<a href="' + revision.restore_url + '" class="button button-primary" target="_blank">Restore This Revision</a>';
                        html += '<a href="' + revision.compare_url + '" class="button button-secondary" target="_blank">Compare with Current</a>';
                        html += '</div>';
                        html += '</div>';
                    });
                }
                
                $('#revisions-content').html(html);
            }
        });
        </script>
        <?php
    }
    
    public function ajax_restore_page() {
        check_ajax_referer('drews_il_restore_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $page_id = intval(wp_unslash($_POST['page_id']));
        
        if (!$page_id) {
            wp_send_json_error('Invalid page ID');
        }
        
        // Get page revisions
        $revisions = wp_get_post_revisions($page_id, array(
            'numberposts' => 10,
            'order' => 'DESC'
        ));
        
        $revision_data = array();
        
        foreach ($revisions as $revision) {
            $revision_data[] = array(
                'id' => $revision->ID,
                'date' => date('M j, Y @ g:i A', strtotime($revision->post_date)),
                'author' => get_the_author_meta('display_name', $revision->post_author),
                'status' => $revision->post_status,
                'changes' => wp_count_posts('revision', 'readable'),
                'restore_url' => admin_url('revision.php?revision=' . $revision->ID),
                'compare_url' => admin_url('revision.php?revision=' . $revision->ID . '&from=' . $page_id)
            );
        }
        
        wp_send_json_success(array(
            'revisions' => $revision_data
        ));
    }
}

// Initialize the emergency restore tool
new DrewsIL_EmergencyRestore();
