jQuery(document).ready(function($) {
    'use strict';
    
    let currentLinkId = null;
    let progressInterval = null;
    
    // Form submission
    $('#drews-il-form').on('submit', function(e) {
        e.preventDefault();
        
        const anchorText = $('#anchor_text').val().trim();
        const targetUrl = $('#target_url').val().trim();
        const scope = $('#scope').val();
        
        if (!anchorText || !targetUrl) {
            showNotice('Please fill in all required fields.', 'error');
            return;
        }
        
        // Show confirmation modal
        showConfirmationModal(
            'Create Internal Link',
            drewsIL.strings.confirm_link.replace('%s', anchorText).replace('%s', targetUrl).replace('%s', getScopeLabel(scope)),
            function() {
                createInternalLink(anchorText, targetUrl, scope);
            }
        );
    });
    
    // Re-run button
    $(document).on('click', '.drews-il-re-run', function() {
        const linkId = $(this).data('link-id');
        const row = $(this).closest('tr');
        const anchorText = row.find('td:first strong').text();
        const targetUrl = row.find('td:nth-child(2) a').attr('href');
        const scope = getScopeFromLabel(row.find('td:nth-child(3)').text());
        
        showConfirmationModal(
            'Re-apply Linking Rule',
            drewsIL.strings.confirm_re_run,
            function() {
                reRunLinking(linkId, anchorText, targetUrl, scope);
            }
        );
    });
    
    // Revert button
    $(document).on('click', '.drews-il-revert', function() {
        const linkId = $(this).data('link-id');
        const row = $(this).closest('tr');
        const anchorText = row.find('td:first strong').text();
        
        showConfirmationModal(
            'Revert Changes',
            `Are you sure you want to revert all changes made by the "${anchorText}" linking rule? This will restore all affected posts to their original state.`,
            function() {
                revertInternalLink(linkId, row);
            }
        );
    });
    
    // Remove button
    $(document).on('click', '.drews-il-remove', function() {
        const linkId = $(this).data('link-id');
        const row = $(this).closest('tr');
        const anchorText = row.find('td:first strong').text();
        
        showConfirmationModal(
            'Remove Internal Link',
            drewsIL.strings.confirm_remove,
            function() {
                removeInternalLink(linkId, row);
            }
        );
    });
    
    // Modal close events
    $('#drews-il-modal-cancel, #drews-il-modal').on('click', function(e) {
        if (e.target === this) {
            hideModal();
        }
    });
    
    // Helper functions
    function getScopeLabel(scope) {
        const labels = {
            'both': 'Pages and Posts',
            'pages': 'Pages Only',
            'posts': 'Posts Only'
        };
        return labels[scope] || scope;
    }
    
    function getScopeFromLabel(label) {
        if (label.includes('Pages & Posts')) return 'both';
        if (label.includes('Pages Only')) return 'pages';
        if (label.includes('Posts Only')) return 'posts';
        return 'both';
    }
    
    function showConfirmationModal(title, message, onConfirm) {
        $('#drews-il-modal-title').text(title);
        $('#drews-il-modal-message').text(message);
        $('#drews-il-modal').show();
        
        // Remove previous event handlers
        $('#drews-il-modal-confirm').off('click');
        
        // Add new event handler
        $('#drews-il-modal-confirm').on('click', function() {
            hideModal();
            onConfirm();
        });
    }
    
    function hideModal() {
        $('#drews-il-modal').hide();
    }
    
    function showNotice(message, type = 'success') {
        const noticeClass = type === 'error' ? 'drews-il-notice error' : 'drews-il-notice success';
        const notice = $('<div class="' + noticeClass + '">' + message + '</div>');
        
        $('.drews-il-container').prepend(notice);
        
        // Auto-remove after 5 seconds
        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    function showProgress() {
        $('#drews-il-progress').show();
        $('.drews-il-submit').addClass('loading');
        $('.drews-il-submit .button-text').text(drewsIL.strings.processing);
    }
    
    function hideProgress() {
        $('#drews-il-progress').hide();
        $('.drews-il-submit').removeClass('loading');
        $('.drews-il-submit .button-text').text('Create Internal Link');
    }
    
    function updateProgress(progress) {
        if (progress.status === 'processing') {
            const percentage = progress.total > 0 ? (progress.current / progress.total) * 100 : 0;
            $('.drews-il-progress-fill').css('width', percentage + '%');
            $('.drews-il-progress-text').text(progress.message);
        } else if (progress.status === 'completed') {
            $('.drews-il-progress-fill').css('width', '100%');
            $('.drews-il-progress-text').text(progress.message);
            
            setTimeout(function() {
                hideProgress();
                if (progress.updated > 0) {
                    showNotice(`Successfully updated ${progress.updated} posts/pages!`, 'success');
                }
            }, 2000);
        }
    }
    
    // AJAX Functions
    function createInternalLink(anchorText, targetUrl, scope) {
        showProgress();
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_create_link',
                nonce: drewsIL.nonce,
                anchor_text: anchorText,
                target_url: targetUrl,
                scope: scope
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data.message, 'success');
                    currentLinkId = response.data.link_id;
                    
                    // Start progress monitoring
                    startProgressMonitoring(currentLinkId);
                    
                    // Reset form
                    $('#drews-il-form')[0].reset();
                    
                    // Reload page after a delay to show new entry
                    setTimeout(function() {
                        location.reload();
                    }, 3000);
                } else {
                    hideProgress();
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                hideProgress();
                showNotice('An error occurred while creating the internal link.', 'error');
            }
        });
    }
    
    function reRunLinking(linkId, anchorText, targetUrl, scope) {
        currentLinkId = linkId;
        showProgress();
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_re_run_link',
                nonce: drewsIL.nonce,
                link_id: linkId
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data, 'success');
                    startProgressMonitoring(linkId);
                } else {
                    hideProgress();
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                hideProgress();
                showNotice('An error occurred while re-applying the linking rule.', 'error');
            }
        });
    }
    
    function revertInternalLink(linkId, row) {
        showProgress();
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_bulk_revert',
                nonce: drewsIL.nonce,
                link_id: linkId
            },
            success: function(response) {
                hideProgress();
                if (response.success) {
                    showNotice(response.data.message, 'success');
                    // Optionally reload the page to show updated data
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                hideProgress();
                showNotice('An error occurred while reverting the internal link changes.', 'error');
            }
        });
    }
    
    function removeInternalLink(linkId, row) {
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_remove_link',
                nonce: drewsIL.nonce,
                link_id: linkId
            },
            success: function(response) {
                if (response.success) {
                    showNotice(response.data, 'success');
                    row.fadeOut(function() {
                        $(this).remove();
                        
                        // Check if table is empty
                        if ($('tbody tr').length === 0) {
                            location.reload();
                        }
                    });
                } else {
                    showNotice(response.data, 'error');
                }
            },
            error: function() {
                showNotice('An error occurred while removing the internal link.', 'error');
            }
        });
    }
    
    function startProgressMonitoring(linkId) {
        if (progressInterval) {
            clearInterval(progressInterval);
        }
        
        progressInterval = setInterval(function() {
                    $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_get_progress',
                nonce: drewsIL.nonce,
                    link_id: linkId
                },
                success: function(response) {
                    if (response.success) {
                        updateProgress(response.data);
                        
                        if (response.data.status === 'completed') {
                            clearInterval(progressInterval);
                            progressInterval = null;
                        }
                    }
                },
                error: function() {
                    clearInterval(progressInterval);
                    progressInterval = null;
                    hideProgress();
                    showNotice('Error checking progress.', 'error');
                }
            });
        }, 1000);
    }
    
    // Quick debug functionality
    $('#quick-debug').on('click', function() {
        $('#quick-debug-modal').show();
        $('#quick-debug-content').html('Searching for "Southbury" in all posts...');
        
        $.ajax({
            url: drewsIL.ajaxurl,
            type: 'POST',
            data: {
                action: 'drews_il_debug_content',
                nonce: drewsIL.nonce,
                search_text: 'Southbury',
                post_types: ['post', 'page']
            },
            success: function(response) {
                if (response.success) {
                    displayQuickDebugResults(response.data);
                } else {
                    $('#quick-debug-content').html('<p style="color: red;">Error: ' + response.data + '</p>');
                }
            },
            error: function() {
                $('#quick-debug-content').html('<p style="color: red;">An error occurred while searching.</p>');
            }
        });
    });
    
    // Close quick debug modal
    $('#quick-debug-close, #quick-debug-modal').on('click', function(e) {
        if (e.target === this) {
            $('#quick-debug-modal').hide();
        }
    });
    
    function displayQuickDebugResults(data) {
        let html = '<h4>Found ' + data.total_matches + ' matches in ' + data.affected_posts + ' posts</h4>';
        
        if (data.posts && data.posts.length > 0) {
            html += '<div style="max-height: 400px; overflow-y: auto;">';
            data.posts.forEach(function(post) {
                html += '<div style="background: #f9f9f9; border: 1px solid #ddd; padding: 10px; margin: 10px 0; border-radius: 4px;">';
                html += '<strong>' + post.post_title + '</strong> (ID: ' + post.post_id + ')<br>';
                html += '<small>Type: ' + post.post_type + ' | Length: ' + post.content_length + ' chars | Matches: ' + post.matches + '</small><br>';
                html += '<div style="background: #fff; border: 1px solid #ccc; padding: 8px; margin: 5px 0; font-family: monospace; font-size: 12px; max-height: 150px; overflow-y: auto;">';
                html += escapeHtml(post.content_preview);
                html += '</div>';
                html += '</div>';
            });
            html += '</div>';
        } else {
            html += '<p style="color: orange;"><strong>No matches found!</strong></p>';
            html += '<p>This means "Southbury" is not found in any post or page content. Possible reasons:</p>';
            html += '<ul>';
            html += '<li>The text might be in the title or excerpt, not the content</li>';
            html += '<li>The text might be encoded differently (HTML entities)</li>';
            html += '<li>The text might be in a different field</li>';
            html += '<li>There might be a database issue</li>';
            html += '</ul>';
        }
        
        $('#quick-debug-content').html(html);
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Clean up on page unload
    $(window).on('beforeunload', function() {
        if (progressInterval) {
            clearInterval(progressInterval);
        }
    });
});
