# ESP Sleek Theme Implementation Guide

## Overview
This guide provides step-by-step instructions for implementing the ESP Sleek theme with Tailwind CSS v4 while maintaining Tailwind v3 for the rest of the application.

## Architecture Overview

### Dual Tailwind Setup
- **ESP Sleek Theme**: Uses Tailwind CSS v4 (Alpha)
- **Rest of Application**: Continues using Tailwind CSS v3
  - Dashboard
  - Filament Admin Panel
  - Other themes

### Why Tailwind v4 for ESP Sleek?
- Modern CSS architecture with @theme directive
- Better performance with native CSS
- Simplified configuration
- Future-proof implementation

## Installation & Setup

### 1. Install Theme Dependencies

Navigate to the theme directory and install dependencies:

```bash
cd themes/ep/esp_sleek
npm install
```

### 2. Build Theme Assets

For development:
```bash
npm run dev
```

For production:
```bash
npm run build
```

### 3. Activate the Theme

Update your `.env` file:
```env
ACTIVE_THEME=esp_sleek
```

### 4. Clear Caches

```bash
php artisan cache:clear
php artisan view:clear
php artisan theme:cache
```

## Theme Structure

```
themes/ep/esp_sleek/
├── composer.json           # Theme metadata
├── package.json           # Node dependencies (Tailwind v4)
├── vite.config.js         # Vite configuration for theme
├── public/                # Compiled assets
│   ├── css/
│   ├── img/
│   └── js/
└── resources/
    ├── css/
    │   └── esp_sleek.css  # Main CSS with Tailwind v4
    ├── js/
    │   ├── esp_theme.js   # Reused from esp_theme
    │   └── google-maps.js # Reused from esp_theme
    └── views/
        ├── layouts/
        ├── components/
        ├── livewire/
        └── pages/
```

## Converting Templates

### 1. Layout Structure

Create base layout at `resources/views/layouts/app.blade.php`:

```blade
<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>@yield('title', config('app.name'))</title>
    
    <!-- Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:ital,opsz,wght@0,14..32,100..900;1,14..32,100..900&display=swap" rel="stylesheet">
    
    <!-- Theme Styles -->
    @vite(['resources/css/esp_sleek.css', 'resources/js/esp_theme.js'], 'themes/ep/esp_sleek/build')
    
    @livewireStyles
    @stack('styles')
</head>
<body class="font-inter">
    <x-frontend.common.header />
    
    <main>
        @yield('content')
    </main>
    
    <x-frontend.common.footer />
    
    @livewireScripts
    <script src="https://cdn.jsdelivr.net/npm/flowbite@2.5.2/dist/flowbite.min.js"></script>
    @stack('scripts')
</body>
</html>
```

### 2. Component Conversion Example

Convert HTML components to Blade components. Example for a member card:

Original HTML:
```html
<div class="border border-solid border-gray-1200 rounded-2xl p-4">
    <h4 class="text-black-1200 font-medium text-2xl">Member Name</h4>
    <!-- ... -->
</div>
```

Blade Component (`resources/views/components/frontend/member/search-card.blade.php`):
```blade
@props(['member'])

<div class="border border-solid border-gray-1200 rounded-2xl p-4 hover:shadow-5xl transition-all duration-300">
    <div class="flex items-start gap-4">
        <div class="relative">
            <div class="relative border border-solid border-gray-1200 rounded-2xl w-20 h-20 p-2.5 flex items-center justify-center">
                <img src="{{ $member->logo_url }}" alt="{{ $member->name }}" class="w-full h-full object-contain">
                @if($member->is_verified)
                    <span class="absolute -right-3.5 -top-3.5 flex items-center justify-center">
                        <img src="{{ theme_asset('img/verify-icon.svg') }}" alt="Verified">
                    </span>
                @endif
            </div>
        </div>
        
        <div class="flex-1">
            <h4 class="text-black-1200 font-medium text-2xl leading-8 flex items-center gap-3">
                {{ $member->name }}
                @if($member->is_pro)
                    <span class="badge-pro">
                        <img src="{{ theme_asset('img/flashlight.svg') }}" alt="Pro"> Pro
                    </span>
                @endif
            </h4>
            
            <p class="font-medium text-sm leading-6 tracking-[-0.176px] text-mid-gray mt-1">
                {{ $member->tagline }}
            </p>
            
            <div class="flex flex-wrap gap-2 mt-3">
                @foreach($member->services as $service)
                    <a href="#" class="tags">{{ $service->name }}</a>
                @endforeach
            </div>
        </div>
    </div>
</div>
```

### 3. Livewire Component Integration

Livewire components work seamlessly with the theme. Example:

```blade
{{-- resources/views/livewire/frontend/member/member-list.blade.php --}}
<div>
    <div class="grid lg:grid-cols-2 grid-cols-1 gap-4">
        @foreach($members as $member)
            <x-frontend.member.search-card :member="$member" />
        @endforeach
    </div>
    
    {{ $members->links('components.frontend.common.pagination') }}
</div>
```

## Tailwind v4 Usage in Theme

### 1. Theme Variables

All theme variables are defined in `@theme` directive in `esp_sleek.css`:

```css
@theme {
    --color-primary-blue: #335cff;
    --color-mid-gray: #525866;
    /* ... other variables ... */
}
```

### 2. Using Theme Colors

```html
<!-- Direct usage -->
<div class="text-primary-blue bg-gray-1400">Content</div>

<!-- With opacity -->
<div class="bg-blue-1400/10">10% opacity blue</div>
```

### 3. Custom Components

Define reusable components in CSS:

```css
.primary-btn {
    @apply flex items-center justify-center font-medium text-sm 
           tracking-[-0.096px] gap-3 bg-primary-blue border 
           border-solid border-primary-blue text-white h-10 
           rounded-full transition-all duration-500 ease-in-out;
}
```

## Development Workflow

### 1. Running Both Build Systems

For full development, run both build processes:

```bash
# Terminal 1: Main app (Tailwind v3)
npm run dev

# Terminal 2: ESP Sleek theme (Tailwind v4)
cd themes/ep/esp_sleek && npm run dev
```

### 2. Asset Compilation

The theme has its own build process separate from the main app:
- Theme assets compile to `themes/ep/esp_sleek/public/`
- Main app assets compile to `public/build/`

### 3. Cache Management

When making changes to theme files:

```bash
php artisan view:clear
php artisan theme:cache
```

## Troubleshooting

### Common Issues

1. **Styles not updating**
   - Clear browser cache
   - Run `php artisan view:clear`
   - Ensure theme build is running

2. **Class not found**
   - Check if using Tailwind v4 syntax in theme
   - Verify class is defined in @theme

3. **JavaScript errors**
   - Ensure Flowbite is loaded
   - Check Alpine.js initialization
   - Verify Livewire is loaded before theme JS

### Debug Commands

```bash
# Check active theme
php artisan theme:list

# Clear all caches
php artisan optimize:clear

# Rebuild theme assets
cd themes/ep/esp_sleek && npm run build
```

## Best Practices

1. **Component Organization**
   - Keep components small and focused
   - Use props for dynamic data
   - Follow naming conventions

2. **Performance**
   - Use lazy loading for images
   - Minimize custom CSS
   - Leverage Tailwind utilities

3. **Maintenance**
   - Document custom components
   - Keep theme assets separate
   - Version control theme builds

## Migration Checklist

When implementing new pages:

- [ ] Create Blade layout if needed
- [ ] Convert HTML to Blade components
- [ ] Update Livewire components
- [ ] Test responsive design
- [ ] Verify all interactions work
- [ ] Check accessibility
- [ ] Test with real data
- [ ] Clear caches and test

## Future Migration to Single Tailwind v4 System

### Migration Timeline
When these become stable:
- **Tailwind CSS v4** (currently alpha)
- **Filament v4** (when it adopts Tailwind v4)
- **Laravel ecosystem v4** updates

### Easy Migration Path

#### 1. Configuration Consolidation
The esp_sleek theme's `@theme` directive can be moved to a global CSS file:

```css
/* resources/css/app.css - Future unified file */
@import "tailwindcss";

@theme {
  /* ESP Sleek colors (keep) */
  --color-primary-blue: #335cff;
  --color-mid-gray: #525866;
  
  /* Add Filament colors */
  --color-primary: #3b82f6;
  --color-gray-50: #f9fafb;
  
  /* Add Dashboard colors */
  --color-dashboard-bg: #f8fafc;
}
```

#### 2. Vite Configuration Merge
```js
// vite.config.js - Future unified version
export default defineConfig({
    plugins: [
        laravel({
            input: [
                'resources/css/app.css',           // Unified Tailwind v4
                'resources/js/app.js',
                'resources/js/dashboard/dashboard.js',
                'themes/ep/esp_sleek/resources/js/esp_theme.js',
                // Remove separate theme CSS, use unified
            ],
            refresh: true,
        }),
        tailwindcss(), // Single Tailwind v4 plugin
    ],
});
```

#### 3. Migration Benefits
**Performance:**
- Single build process - Faster compilation
- Shared CSS bundle - Smaller file sizes
- Unified caching - Better browser cache utilization

**Maintenance:**
- One Tailwind config - Easier color/theme management
- Consistent versioning - No dependency conflicts
- Simplified deployment - Single build command

**Development Experience:**
- Hot reload for all - Changes reflect everywhere instantly
- Shared design tokens - Consistent styling across app
- Easier debugging - Single source of truth

### Migration Checklist (Future)
- [ ] Verify Tailwind v4 and Filament v4 stability
- [ ] Backup current dual-system setup
- [ ] Test compatibility across all interfaces
- [ ] Update dependencies to v4 stable
- [ ] Merge @theme configurations
- [ ] Consolidate Vite configs
- [ ] Test thoroughly - all interfaces and components
- [ ] Deploy gradually - staging → production

## Support

For issues or questions:
1. Check Laravel Themes Manager docs
2. Review Tailwind v4 documentation
3. Check Livewire compatibility
4. Test in isolation first