<?php

namespace App\Console\Commands;

use App\Models\Setting;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class CleanupSettingsFiles extends Command
{
    protected $signature = 'settings:cleanup-files';
    protected $description = 'Clean up orphaned settings files that are no longer referenced in database';

    public function handle()
    {
        $this->info('Cleaning up orphaned settings files...');

        $disk = Storage::disk('public');
        $settingsDirectory = 'settings';

        if (!$disk->exists($settingsDirectory)) {
            $this->info('Settings directory does not exist.');
            return Command::SUCCESS;
        }

        // Get all files in the settings directory
        $allFiles = $disk->allFiles($settingsDirectory);
        $this->info('Found ' . count($allFiles) . ' files in settings directory.');

        // Get all file settings from database
        $fileSettings = Setting::whereIn('type', ['file', 'image', 'document'])
            ->whereNotNull('value')
            ->pluck('value')
            ->filter()
            ->toArray();

        $this->info('Found ' . count($fileSettings) . ' file settings in database.');

        // Extract file paths from settings values
        $referencedFiles = [];
        foreach ($fileSettings as $settingValue) {
            $filePath = $settingValue;

            // Handle full URLs by extracting relative path
            if (str_contains($filePath, '/storage/')) {
                $filePath = str_replace('/storage/', '', parse_url($filePath, PHP_URL_PATH));
            }

            // Remove leading slash if present
            $filePath = ltrim($filePath, '/');

            if (!empty($filePath)) {
                $referencedFiles[] = $filePath;
            }
        }

        $this->info('Found ' . count($referencedFiles) . ' referenced files.');

        // Find orphaned files
        $orphanedFiles = array_diff($allFiles, $referencedFiles);

        if (empty($orphanedFiles)) {
            $this->info('No orphaned files found.');
            return Command::SUCCESS;
        }

        $this->info('Found ' . count($orphanedFiles) . ' orphaned files:');

        foreach ($orphanedFiles as $orphanedFile) {
            $this->line("- {$orphanedFile}");
        }

        if ($this->confirm('Do you want to delete these orphaned files?')) {
            $deletedCount = 0;

            foreach ($orphanedFiles as $orphanedFile) {
                try {
                    $disk->delete($orphanedFile);
                    $deletedCount++;
                    $this->line("Deleted: {$orphanedFile}");
                } catch (\Exception $e) {
                    $this->error("Failed to delete: {$orphanedFile} - " . $e->getMessage());
                }
            }

            $this->info("Successfully deleted {$deletedCount} orphaned files.");
        } else {
            $this->info('Cleanup cancelled.');
        }

        return Command::SUCCESS;
    }
}
