<?php

declare(strict_types=1);

namespace App\Console\Commands;

use App\Models\User\User;
use App\Models\Membership\MembershipPlan;
use App\Models\Membership\Subscription;
use App\Enums\MembershipPaymentTerm;
use App\Enums\SubscriptionStatus;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class CreateServiceUserSubscriptions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'subscriptions:create-for-service-users 
                            {--dry-run : Show what would be created without actually creating subscriptions}
                            {--force : Skip confirmation prompts}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create subscriptions for all users with Service role using random membership plans';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Starting subscription creation for Service users...');

        // Get all users with Service role
        $serviceUsers = User::role('Service')
            ->whereDoesntHave('subscriptions', function ($query) {
                $query->where('status', SubscriptionStatus::Active);
            })
            ->get();

        if ($serviceUsers->isEmpty()) {
            $this->info('No Service users found without active subscriptions.');
            return self::SUCCESS;
        }

        // Get all active membership plans
        $membershipPlans = MembershipPlan::where('is_active', true)
            ->get();

        if ($membershipPlans->isEmpty()) {
            $this->error('No active membership plans found. Cannot create subscriptions.');
            return self::FAILURE;
        }

        $this->info("Found {$serviceUsers->count()} Service users without active subscriptions.");
        $this->info("Found {$membershipPlans->count()} available membership plans.");

        if ($this->option('dry-run')) {
            $this->warn('DRY RUN MODE - No subscriptions will be created.');
            $this->showDryRunPreview($serviceUsers, $membershipPlans);
            return self::SUCCESS;
        }

        // Ask for confirmation unless --force is used
        if (!$this->option('force') && !$this->confirm('Do you want to proceed with creating subscriptions?')) {
            $this->info('Operation cancelled.');
            return self::SUCCESS;
        }

        return $this->createSubscriptions($serviceUsers, $membershipPlans);
    }

    /**
     * Show a preview of what would be created in dry-run mode
     */
    private function showDryRunPreview($users, $plans): void
    {
        $this->table(
            ['User ID', 'User Name', 'Email', 'Random Plan Selected'],
            $users->take(10)->map(function ($user) use ($plans) {
                $randomPlan = $plans->random();
                return [
                    $user->id,
                    $user->name,
                    $user->email,
                    $randomPlan->package_title
                ];
            })->toArray()
        );

        if ($users->count() > 10) {
            $this->info("... and " . ($users->count() - 10) . " more users");
        }
    }

    /**
     * Create subscriptions for the users
     */
    private function createSubscriptions($users, $plans): int
    {
        $bar = $this->output->createProgressBar($users->count());
        $bar->start();

        $successCount = 0;
        $errorCount = 0;
        $errors = [];

        foreach ($users as $user) {
            DB::beginTransaction();
            try {
                // Select a random membership plan
                $randomPlan = $plans->random();

                // Determine payment term based on plan's preferred term or random
                $paymentTerm = $randomPlan->preferred_payment_term ?? $this->getRandomPaymentTerm();

                // Calculate subscription period
                $currentPeriodStart = Carbon::now();
                $currentPeriodEnd = $this->calculatePeriodEnd($currentPeriodStart, $paymentTerm);

                $subscription = Subscription::create([
                    'uuid' => (string) Str::uuid(),
                    'user_id' => $user->id,
                    'membership_plan_id' => $randomPlan->id,
                    'purchase_date' => $currentPeriodStart,
                    'payment_term' => $paymentTerm,
                    'is_active' => true,
                    'auto_renewal' => true,
                    'current_period_start' => $currentPeriodStart,
                    'current_period_end' => $currentPeriodEnd,
                    'status' => SubscriptionStatus::Active,
                    'created_by' => 1, // System user - you may want to adjust this
                ]);

                DB::commit();
                $successCount++;
            } catch (\Exception $e) {
                DB::rollBack();
                $errorCount++;
                $errors[] = "User {$user->id} ({$user->email}): {$e->getMessage()}";
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        // Display results
        $this->info("Successfully created {$successCount} subscriptions.");

        if ($errorCount > 0) {
            $this->error("Failed to create {$errorCount} subscriptions:");
            foreach ($errors as $error) {
                $this->error("  - {$error}");
            }
        }

        // Show summary table
        if ($successCount > 0) {
            $this->info("\nSubscription Summary:");
            $recentSubscriptions = Subscription::with(['user', 'membershipPlan'])
                ->whereIn('user_id', $users->pluck('id'))
                ->latest()
                ->take(5)
                ->get();

            $this->table(
                ['User', 'Email', 'Plan', 'Payment Term', 'Period End'],
                $recentSubscriptions->map(function ($sub) {
                    return [
                        $sub->user->name,
                        $sub->user->email,
                        $sub->membershipPlan->package_title,
                        $sub->payment_term->getLabel(),
                        $sub->current_period_end->format('Y-m-d')
                    ];
                })->toArray()
            );

            if ($successCount > 5) {
                $this->info("... and " . ($successCount - 5) . " more subscriptions created.");
            }
        }

        return $errorCount > 0 ? self::FAILURE : self::SUCCESS;
    }

    /**
     * Get a random payment term
     */
    private function getRandomPaymentTerm(): MembershipPaymentTerm
    {
        $terms = [
            MembershipPaymentTerm::Monthly,
            MembershipPaymentTerm::Quarterly,
            MembershipPaymentTerm::SemiYearly,
            MembershipPaymentTerm::Yearly,
        ];

        return $terms[array_rand($terms)];
    }

    /**
     * Calculate the end date of the subscription period based on payment term
     */
    private function calculatePeriodEnd(Carbon $startDate, MembershipPaymentTerm $paymentTerm): Carbon
    {
        return match ($paymentTerm) {
            MembershipPaymentTerm::Monthly => $startDate->copy()->addMonth(),
            MembershipPaymentTerm::Quarterly => $startDate->copy()->addMonths(3),
            MembershipPaymentTerm::SemiYearly => $startDate->copy()->addMonths(6),
            MembershipPaymentTerm::Yearly => $startDate->copy()->addYear(),
        };
    }
}
