<?php

namespace App\Console\Commands;

use App\Models\User\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class GenerateUserUuids extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'users:generate-uuids {--chunk=100 : Number of users to process per chunk}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate and set missing UUID values for all users';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $chunkSize = $this->option('chunk');

        $this->info('Starting UUID generation for users...');

        $totalUsers = User::whereNull('uuid')->orWhere('uuid', '')->count();

        if ($totalUsers === 0) {
            $this->info('All users already have UUIDs assigned.');
            return self::SUCCESS;
        }

        $this->info("Found {$totalUsers} users without UUIDs.");
        $bar = $this->output->createProgressBar($totalUsers);
        $bar->start();

        $processedCount = 0;

        // Process in chunks to avoid memory issues with large datasets
        User::whereNull('uuid')
            ->orWhere('uuid', '')
            ->chunkById($chunkSize, function ($users) use (&$processedCount, $bar) {
                DB::beginTransaction();
                try {
                    foreach ($users as $user) {
                        $user->uuid = (string) Str::uuid();
                        $user->save();
                        $processedCount++;
                        $bar->advance();
                    }
                    DB::commit();
                } catch (\Exception $e) {
                    DB::rollBack();
                    $this->error("Error processing users: {$e->getMessage()}");
                    throw $e;
                }
            });

        $bar->finish();
        $this->newLine(2);
        $this->info("Successfully generated UUIDs for {$processedCount} users.");

        return self::SUCCESS;
    }
}
