<?php

namespace App\Console\Commands;

use App\Enums\UserProfileType;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\User\User;
use App\Models\User\UserProfile;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ImportCompaniesCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:import-companies';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import companies from the markdown file and create users with profiles and locations';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Importing companies from markdown file...');

        // Path to the markdown file
        $filePath = public_path('sample-data/companies.md');

        if (!File::exists($filePath)) {
            $this->error('Companies markdown file not found at: ' . $filePath);
            return 1;
        }

        // Read the file content
        $content = File::get($filePath);

        // Parse the markdown table
        $companies = $this->parseMarkdownTable($content);

        if (empty($companies)) {
            $this->error('No companies found in the markdown file.');
            return 1;
        }

        $this->info('Found ' . count($companies) . ' companies.');

        // Find or create Canada country
        $canada = Country::firstOrCreate(
            ['country_name' => 'Canada'],
            ['country_code' => 'CA', 'uuid' => Str::uuid()]
        );

        // Create a progress bar
        $progressBar = $this->output->createProgressBar(count($companies));
        $progressBar->start();

        DB::beginTransaction();

        try {
            foreach ($companies as $index => $company) {
                // Skip header row if it exists
                if (
                    $index === 0 && isset($company['Company Name']) &&
                    isset($company['Latitude']) && isset($company['Longitude'])
                ) {
                    continue;
                }

                $companyName = trim($company['Company Name'] ?? '');
                $latitude = trim($company['Latitude'] ?? '');
                $longitude = trim($company['Longitude'] ?? '');
                $cityName = trim($company['City'] ?? '');
                $stateName = trim($company['State/Province'] ?? '');

                if (
                    empty($companyName) || empty($latitude) || empty($longitude) ||
                    empty($cityName) || empty($stateName)
                ) {
                    continue;
                }

                // Create user with factory
                $user = User::factory()->create();

                // Assign Service role to the user
                $user->assignRole('Service');

                // Find or create state
                $state = State::firstOrCreate(
                    ['state_name' => $stateName, 'country_id' => $canada->id],
                    ['uuid' => Str::uuid()]
                );

                // Find or create city
                $city = City::firstOrCreate(
                    ['city_name' => $cityName, 'state_id' => $state->id, 'country_id' => $canada->id],
                    ['uuid' => Str::uuid()]
                );

                // Create user profile with company name
                $profile = UserProfile::factory()->make([
                    'company_name' => $companyName,
                    'profile_type' => UserProfileType::Business,
                ]);

                $user->profile()->save($profile);

                // Create user location with coordinates
                $user->locations()->create([
                    'uuid' => Str::uuid(),
                    'location_name' => $companyName . ' Headquarters',
                    'contact_person_name' => fake()->name(),
                    'phone' => fake()->phoneNumber(),
                    'email' => fake()->safeEmail(),
                    'fax' => fake()->phoneNumber(),
                    'map_address' => "$companyName, $cityName, $stateName, Canada",
                    'address_line_1' => fake()->streetAddress(),
                    'address_line_2' => fake()->secondaryAddress(),
                    'postal_code' => fake()->postcode(),
                    'latitude' => $latitude,
                    'longitude' => $longitude,
                    'is_active' => true,
                    'country_id' => $canada->id,
                    'state_id' => $state->id,
                    'city_id' => $city->id,
                    'primary_location' => true,
                    'created_by' => $user->id,
                ]);

                $progressBar->advance();
            }

            DB::commit();
            $progressBar->finish();

            $this->newLine();
            $this->info('Companies imported successfully!');

            return 0;
        } catch (\Exception $e) {
            DB::rollBack();

            $this->newLine();
            $this->error('Error importing companies: ' . $e->getMessage());

            return 1;
        }
    }

    /**
     * Parse a markdown table into an array of records
     */
    private function parseMarkdownTable(string $content): array
    {
        $lines = explode("\n", $content);

        // Need at least 3 lines for a table (header, separator, data)
        if (count($lines) < 3) {
            return [];
        }

        // Extract headers
        $headerLine = trim($lines[0], "| \t\n\r\0\x0B");
        $headers = array_map('trim', explode('|', $headerLine));

        // Skip separator line

        $records = [];

        // Start from line 2 (after header and separator)
        for ($i = 2; $i < count($lines); $i++) {
            $line = trim($lines[$i], "| \t\n\r\0\x0B");

            // Skip empty lines
            if (empty($line)) {
                continue;
            }

            $values = array_map('trim', explode('|', $line));

            // If number of values doesn't match number of headers, skip
            if (count($values) !== count($headers)) {
                continue;
            }

            $record = array_combine($headers, $values);
            $records[] = $record;
        }

        return $records;
    }
}
