<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Blog\Post;
use App\Enums\PostStatus;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Filament\Notifications\Notification;
class PublishScheduledPosts extends Command
{
    protected $signature = 'posts:publish-scheduled';
    protected $description = 'Publish scheduled posts that have reached their publication date';

    public function handle(): void
    {
        // Use Cache facade instead of Redis for locking
        $lock = Cache::lock('publishing_scheduled_posts', 60);

        if ($lock->get()) {
            try {
                Post::scheduledForPublishing()
                    ->chunk(100, function ($posts) {
                        foreach ($posts as $post) {
                            $this->publishPost($post);
                        }
                    });
            } finally {
                $lock->release();
            }
        }
    }

    private function publishPost(Post $post): void
    {
        try {
            $post->update([
                'post_status' => PostStatus::Published,
            ]);

            $this->info("Published post: {$post->post_title} (ID: {$post->id})");
            Log::info("Scheduled post published: {$post->post_title} (ID: {$post->id})");
            Notification::make()
                ->title('Scheduled post published')
                ->body("The post {$post->post_title} (ID: {$post->id}) has been published.")
                ->success()
                ->color('success')
                ->sendToDatabase($post->author);

        } catch (\Exception $e) {

            $this->error("Failed to publish post {$post->id}: {$e->getMessage()}");
            Log::error("Failed to publish scheduled post {$post->id}: {$e->getMessage()}");
            Notification::make()
                ->title('Failed to publish scheduled post')
                ->body("The post {$post->post_title} (ID: {$post->id}) failed to publish.")
                ->danger()
                ->color('danger')
                ->sendToDatabase($post->author);
        }
    }
} 