<?php

namespace App\Console\Commands;

use App\Enums\JobStatus;
use App\Enums\JobWorkMode;
use App\Models\Job\Job;
use App\Models\Job\JobRole;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class UpdateJobsDataCommand extends Command
{
    protected $signature = 'jobs:update-data {--dry-run : Run without making changes}';
    protected $description = 'Update jobs with missing data and fix relationships';

    public function handle(): int
    {
        $this->info('Starting job data update process...');

        // Check if this is a dry run
        $isDryRun = $this->option('dry-run');
        if ($isDryRun) {
            $this->warn('Running in DRY RUN mode - no changes will be saved to database');
        }

        // Get total count for progress bar
        $totalJobs = Job::count();
        $bar = $this->output->createProgressBar($totalJobs);
        $bar->start();

        // Counters for summary
        $updatedLocationCount = 0;
        $updatedRoleCount = 0;
        $updatedWorkModeCount = 0;
        $updatedStatusCount = 0;

        if (!$isDryRun) {
            DB::beginTransaction();
        }

        try {
            // Process each job
            Job::chunk(100, function ($jobs) use (
                &$updatedLocationCount,
                &$updatedRoleCount,
                &$updatedWorkModeCount,
                &$updatedStatusCount,
                $bar,
                $isDryRun
            ) {

                foreach ($jobs as $job) {
                    // 1. Update location data if missing
                    if (empty($job->country_id) || empty($job->state_id) || empty($job->city_id)) {
                        // Try to determine location from user's data if available
                        if ($job->user && $job->user->country_id) {
                            $job->country_id = $job->user->country_id;
                            $job->state_id = $job->user->state_id;
                            $job->city_id = $job->user->city_id;
                            $updatedLocationCount++;
                        }
                    }

                    // 2. Handle job role relationship - remove and reattach with child role
                    if ($job->job_role_id) {
                        $currentRole = JobRole::find($job->job_role_id);

                        // If current role is a parent, find a suitable child or keep parent as fallback
                        if ($currentRole && $currentRole->parent_id === null) {
                            // Get first child or keep parent if no children
                            $childRole = $currentRole->children()->first();

                            if ($childRole) {
                                $job->job_role_id = $childRole->id;
                                $updatedRoleCount++;
                            }
                        }
                    }

                    // 3. Fill work_mode if empty
                    if (empty($job->work_mode)) {
                        // Default to Onsite if not specified
                        $job->work_mode = JobWorkMode::Onsite->value;
                        $updatedWorkModeCount++;
                    }

                    // 4. Change status from draft to published
                    if ($job->job_status === JobStatus::Draft) {
                        $job->job_status = JobStatus::Published;

                        // Set published_at if not already set
                        if (empty($job->published_at)) {
                            $job->published_at = now();
                        }

                        $updatedStatusCount++;
                    }

                    // Save changes only if not a dry run
                    if (!$isDryRun) {
                        $job->save();
                    }

                    // Update progress bar
                    $bar->advance();
                }
            });

            if (!$isDryRun) {
                DB::commit();
            }

            $bar->finish();

            // Show summary
            $this->newLine(2);
            $this->info('Job data update ' . ($isDryRun ? 'simulation' : 'process') . ' completed successfully!');
            $this->table(
                ['Updated Field', 'Count'],
                [
                    ['Location Data', $updatedLocationCount],
                    ['Job Roles', $updatedRoleCount],
                    ['Work Mode', $updatedWorkModeCount],
                    ['Status (Draft → Published)', $updatedStatusCount],
                ]
            );

            if ($isDryRun) {
                $this->warn('This was a dry run. Run without --dry-run option to apply these changes.');
            }

            return Command::SUCCESS;
        } catch (\Exception $e) {
            if (!$isDryRun) {
                DB::rollBack();
            }

            $this->error("An error occurred: {$e->getMessage()}");
            $this->error($e->getTraceAsString());

            return Command::FAILURE;
        }
    }
}
