<?php

declare(strict_types=1);

namespace App\Console\Commands;

use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\User\UserLocation;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Symfony\Component\Console\Command\Command as CommandAlias;

class UpdateUserLocationsFromJsonCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:update-user-locations-from-csv';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update user locations (IDs 203-410) with data from Canada_Demo_Locations__200_.csv, creating cities and states as needed.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Starting user location update process from CSV...');

        $csvPath = public_path('sample-data/Canada_Demo_Locations__200_.csv');

        if (!File::exists($csvPath)) {
            $this->error('Canada_Demo_Locations__200_.csv file not found in public/sample-data/');
            return CommandAlias::FAILURE;
        }

        // Read the CSV file
        $csvContent = file($csvPath);
        if (empty($csvContent)) {
            $this->error('No data found in CSV file or data is invalid.');
            return CommandAlias::FAILURE;
        }

        // Get the header from the first line
        $header = str_getcsv(array_shift($csvContent));

        // Parse the remaining CSV data
        $locationsData = [];
        foreach ($csvContent as $line) {
            $row = str_getcsv($line);
            if (count($header) !== count($row)) {
                $this->warn("Skipping CSV row due to column count mismatch: " . implode(',', $row));
                continue;
            }
            $locationsData[] = array_combine($header, $row);
        }

        if (empty($locationsData)) {
            $this->error('No valid data rows found in CSV after processing headers.');
            return CommandAlias::FAILURE;
        }

        $userLocationsToUpdate = UserLocation::whereBetween('id', [1, 200])->get();

        if ($userLocationsToUpdate->isEmpty()) {
            $this->warn('No user locations found with IDs between 203 and 410. Nothing to update.');
            return CommandAlias::SUCCESS;
        }

        $this->info("Found {$userLocationsToUpdate->count()} user locations to update.");

        DB::transaction(function () use ($userLocationsToUpdate, $locationsData) {
            $locationDataCount = count($locationsData);
            $locationDataIndex = 0;

            $country = Country::firstOrCreate(['country_name' => 'Canada']);

            foreach ($userLocationsToUpdate as $userLocation) {
                $currentCsvLocation = $locationsData[$locationDataIndex % $locationDataCount];

                $state = State::firstOrCreate(
                    ['state_name' => $currentCsvLocation['province'], 'country_id' => $country->id],
                    ['country_id' => $country->id]
                );

                $city = City::firstOrCreate(
                    ['city_name' => $currentCsvLocation['city'], 'state_id' => $state->id],
                    ['state_id' => $state->id]
                );

                $userLocation->latitude = (float) $currentCsvLocation['lat'];
                $userLocation->longitude = (float) $currentCsvLocation['lng'];
                $userLocation->city_id = $city->id;
                $userLocation->state_id = $state->id;
                $userLocation->country_id = $country->id;
                $userLocation->map_address = $currentCsvLocation['city'] . ', ' . $currentCsvLocation['province'] . ', Canada';

                $userLocation->save();

                $this->output->writeln("Updated UserLocation ID: {$userLocation->id} with City: {$city->name}, State: {$state->name}");

                $locationDataIndex++;
            }
        });

        $this->info('User locations updated successfully from CSV.');
        return CommandAlias::SUCCESS;
    }
}
