<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User\User;
use App\Services\Review\ReviewService;

class UpdateUserProfileRatings extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'users:update-profile-ratings';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update profile ratings for all existing users based on their reviews';

    protected ReviewService $reviewService;

    public function __construct(ReviewService $reviewService)
    {
        parent::__construct();
        $this->reviewService = $reviewService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to update user profile ratings...');
        
        $totalUsers = User::whereHas('profile')->count();
        $this->info("Found {$totalUsers} users with profiles to update.");
        
        $bar = $this->output->createProgressBar($totalUsers);
        $bar->start();
        
        $updatedCount = 0;
        
        User::whereHas('profile')
            ->chunk(100, function ($users) use ($bar, &$updatedCount) {
                foreach ($users as $user) {
                    // Calculate average rating for the user
                    $averageRating = $this->reviewService->getAverageRating(User::class, $user->id);
                    
                    // Update profile rating
                    if ($user->profile) {
                        $user->profile->update([
                            'profile_rating' => $averageRating
                        ]);
                        $updatedCount++;
                    }
                    
                    $bar->advance();
                }
            });
        
        $bar->finish();
        $this->newLine();
        
        $this->info("Successfully updated {$updatedCount} user profile ratings.");
        
        return Command::SUCCESS;
    }
}
