<?php

namespace App\DTOs\ApiClient;

use App\Enums\ApiClientType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class CreateApiClientDTO
{
    public string $name;
    public ApiClientType $type;
    public bool $is_active;
    public ?int $created_by;

    public function __construct(array $data)
    {
        $this->name = $data['name'];
        $this->type = isset($data['type'])
            ? (is_string($data['type']) ? ApiClientType::from($data['type']) : $data['type'])
            : ApiClientType::Website;
        $this->is_active = $data['is_active'] ?? true;
        $this->created_by = $data['created_by'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'name' => 'required|string|max:255',
            'type' => [
                'sometimes',
                'string',
                Rule::in(ApiClientType::values())
            ],
            'is_active' => 'sometimes|boolean',
            'created_by' => 'nullable|integer|exists:users,id',
        ];
    }

    public function toArray(): array
    {
        return [
            'name' => $this->name,
            'type' => $this->type->value,
            'is_active' => $this->is_active,
            'created_by' => $this->created_by,
        ];
    }
}
