<?php

namespace App\DTOs\ApiClient;

use App\Enums\ApiClientType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class UpdateApiClientDTO
{
    public ?string $name;
    public ?ApiClientType $type;
    public ?bool $is_active;

    public function __construct(array $data)
    {
        $this->name = $data['name'] ?? null;
        $this->type = isset($data['type'])
            ? (is_string($data['type']) ? ApiClientType::from($data['type']) : $data['type'])
            : null;
        $this->is_active = $data['is_active'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'name' => 'sometimes|string|max:255',
            'type' => [
                'sometimes',
                'string',
                Rule::in(ApiClientType::values())
            ],
            'is_active' => 'sometimes|boolean',
        ];
    }

    public function toArray(): array
    {
        $data = [];

        if ($this->name !== null) {
            $data['name'] = $this->name;
        }
        if ($this->type !== null) {
            $data['type'] = $this->type->value;
        }
        if ($this->is_active !== null) {
            $data['is_active'] = $this->is_active;
        }

        return $data;
    }

    public function hasChanges(): bool
    {
        return !empty($this->toArray());
    }
}
