<?php

namespace App\DTOs\Lead\Lead;

use App\Enums\LeadStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class CreateLeadDTO
{
    public ?int $lead_form_id;
    public ?array $lead_data;
    public ?string $lead_source;
    public ?string $first_name;
    public ?string $last_name;
    public ?string $email;
    public ?string $phone;
    public ?string $message;
    public ?LeadStatus $lead_status;
    public ?int $created_by;

    // Optional user assignment
    public ?array $assigned_user_ids;
    public ?array $user_statuses; // ['user_id' => 'status']

    // Optional metadata
    public ?array $meta;

    public function __construct(array $data)
    {
        $this->lead_form_id = $data['lead_form_id'] ?? null;
        $this->lead_data = $data['lead_data'] ?? null;
        $this->lead_source = $data['lead_source'] ?? null;
        $this->first_name = $data['first_name'] ?? null;
        $this->last_name = $data['last_name'] ?? null;
        $this->email = $data['email'] ?? null;
        $this->phone = $data['phone'] ?? null;
        $this->message = $data['message'] ?? null;
        $this->lead_status = isset($data['lead_status'])
            ? LeadStatus::from($data['lead_status'])
            : LeadStatus::Pending;
        $this->created_by = $data['created_by'] ?? null;
        $this->assigned_user_ids = $data['assigned_user_ids'] ?? null;
        $this->user_statuses = $data['user_statuses'] ?? null;
        $this->meta = $data['meta'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'lead_form_id' => 'nullable|integer|exists:lead_forms,id',
            'lead_data' => 'nullable|array',
            'lead_data.*' => 'nullable',
            'lead_source' => 'nullable|string|max:255',
            'first_name' => 'nullable|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'message' => 'nullable|string|max:5000',
            'lead_status' => [
                'nullable',
                'string',
                Rule::in(array_column(LeadStatus::cases(), 'value'))
            ],
            'created_by' => 'nullable|integer|exists:users,id',
            'assigned_user_ids' => 'nullable|array',
            'assigned_user_ids.*' => 'integer|exists:users,id',
            'user_statuses' => 'nullable|array',
            'user_statuses.*' => Rule::in(array_column(LeadStatus::cases(), 'value')),
            'meta' => 'nullable|array',
        ];
    }

    public function toArray(): array
    {
        $data = [
            'lead_form_id' => $this->lead_form_id,
            'lead_status' => $this->lead_status?->value,
        ];

        if ($this->lead_data !== null) {
            $data['lead_data'] = $this->lead_data;
        }
        if ($this->lead_source !== null) {
            $data['lead_source'] = $this->lead_source;
        }
        if ($this->first_name !== null) {
            $data['first_name'] = $this->first_name;
        }
        if ($this->last_name !== null) {
            $data['last_name'] = $this->last_name;
        }
        if ($this->email !== null) {
            $data['email'] = $this->email;
        }
        if ($this->phone !== null) {
            $data['phone'] = $this->phone;
        }
        if ($this->message !== null) {
            $data['message'] = $this->message;
        }
        if ($this->created_by !== null) {
            $data['created_by'] = $this->created_by;
        }

        return $data;
    }

    public function getAssignedUsers(): ?array
    {
        return $this->assigned_user_ids;
    }

    public function getUserStatuses(): ?array
    {
        return $this->user_statuses;
    }

    public function getMeta(): ?array
    {
        return $this->meta;
    }
}
