<?php

namespace App\DTOs\Lead\Lead;

use App\Models\Lead\Lead;
use App\Enums\LeadStatus;

class LeadDTO
{
    public int $id;
    public string $uuid;
    public int $lead_form_id;
    public ?array $lead_data;
    public ?string $lead_source;
    public ?string $first_name;
    public ?string $last_name;
    public ?string $email;
    public ?string $phone;
    public ?string $message;
    public ?LeadStatus $lead_status;
    public ?int $created_by;
    public string $created_at;
    public string $updated_at;
    public ?string $deleted_at;

    // Relationships
    public ?array $lead_form;
    public ?array $form_fields;
    public ?array $users;
    public ?array $author;
    public ?array $meta;

    public function __construct(Lead $lead, bool $includeRelations = false)
    {
        $this->id = $lead->id;
        $this->uuid = $lead->uuid;
        $this->lead_form_id = $lead->lead_form_id;
        $this->lead_data = $lead->lead_data;
        $this->lead_source = $lead->lead_source;
        $this->first_name = $lead->first_name;
        $this->last_name = $lead->last_name;
        $this->email = $lead->email;
        $this->phone = $lead->phone;
        $this->message = $lead->message;
        $this->lead_status = $lead->lead_status;
        $this->created_by = $lead->created_by;
        $this->created_at = $lead->created_at->toIso8601String();
        $this->updated_at = $lead->updated_at->toIso8601String();
        $this->deleted_at = $lead->deleted_at?->toIso8601String();

        if ($includeRelations) {
            $this->lead_form = $lead->leadForm ? [
                'id' => $lead->leadForm->id,
                'title' => $lead->leadForm->title,
                'uuid' => $lead->leadForm->uuid,
            ] : null;

            $this->form_fields = $lead->formFields?->map(fn($field) => [
                'id' => $field->id,
                'field_name' => $field->field_name,
                'field_type' => $field->field_type,
            ])->toArray();

            $this->users = $lead->users?->map(fn($user) => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'lead_status' => $user->pivot->lead_status ?? null,
            ])->toArray();

            $this->author = $lead->author ? [
                'id' => $lead->author->id,
                'name' => $lead->author->name,
                'email' => $lead->author->email,
            ] : null;

            $this->meta = $lead->getAllMeta();
        }
    }

    public function toArray(): array
    {
        $data = [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'lead_form_id' => $this->lead_form_id,
            'lead_data' => $this->lead_data,
            'lead_source' => $this->lead_source,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'email' => $this->email,
            'phone' => $this->phone,
            'message' => $this->message,
            'lead_status' => $this->lead_status?->value,
            'created_by' => $this->created_by,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            'deleted_at' => $this->deleted_at,
        ];

        if (isset($this->lead_form)) {
            $data['lead_form'] = $this->lead_form;
        }
        if (isset($this->form_fields)) {
            $data['form_fields'] = $this->form_fields;
        }
        if (isset($this->users)) {
            $data['users'] = $this->users;
        }
        if (isset($this->author)) {
            $data['author'] = $this->author;
        }
        if (isset($this->meta)) {
            $data['meta'] = $this->meta;
        }

        return $data;
    }

    public function getFullName(): string
    {
        return trim(($this->first_name ?? '') . ' ' . ($this->last_name ?? ''));
    }
}
