<?php

namespace App\DTOs\Lead\Lead;

use App\Enums\LeadStatus;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class LeadFilterDTO
{
    public ?int $lead_form_id;
    public ?string $lead_source;
    public ?array $lead_statuses; // Multiple statuses
    public ?string $search; // Search in first_name, last_name, email, phone, message
    public ?string $email;
    public ?string $phone;
    public ?int $created_by;
    public ?int $assigned_to_user_id;

    // Date filtering
    public ?string $created_from; // Y-m-d format
    public ?string $created_to;
    public ?string $updated_from;
    public ?string $updated_to;

    // Pagination & sorting
    public int $page;
    public int $per_page;
    public string $order_by;
    public string $order_direction;

    // Include soft deleted
    public bool $with_trashed;
    public bool $only_trashed;

    // Include relationships
    public bool $with_lead_form;
    public bool $with_form_fields;
    public bool $with_users;
    public bool $with_author;
    public bool $with_meta;

    public function __construct(array $data)
    {
        $this->lead_form_id = $data['lead_form_id'] ?? null;
        $this->lead_source = $data['lead_source'] ?? null;
        $this->lead_statuses = isset($data['lead_statuses'])
            ? array_map(fn($status) => LeadStatus::from($status), (array)$data['lead_statuses'])
            : null;
        $this->search = $data['search'] ?? null;
        $this->email = $data['email'] ?? null;
        $this->phone = $data['phone'] ?? null;
        $this->created_by = $data['created_by'] ?? null;
        $this->assigned_to_user_id = $data['assigned_to_user_id'] ?? null;

        // Date filters
        $this->created_from = $data['created_from'] ?? null;
        $this->created_to = $data['created_to'] ?? null;
        $this->updated_from = $data['updated_from'] ?? null;
        $this->updated_to = $data['updated_to'] ?? null;

        // Pagination & sorting
        $this->page = $data['page'] ?? 1;
        $this->per_page = $data['per_page'] ?? 15;
        $this->order_by = $data['order_by'] ?? 'created_at';
        $this->order_direction = $data['order_direction'] ?? 'desc';

        // Soft deletes
        $this->with_trashed = $data['with_trashed'] ?? false;
        $this->only_trashed = $data['only_trashed'] ?? false;

        // Relationships
        $this->with_lead_form = $data['with_lead_form'] ?? false;
        $this->with_form_fields = $data['with_form_fields'] ?? false;
        $this->with_users = $data['with_users'] ?? false;
        $this->with_author = $data['with_author'] ?? false;
        $this->with_meta = $data['with_meta'] ?? false;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate([
            'lead_form_id' => 'sometimes|integer|exists:lead_forms,id',
            'lead_source' => 'sometimes|string|max:255',
            'lead_statuses' => 'sometimes|array',
            'lead_statuses.*' => Rule::in(array_column(LeadStatus::cases(), 'value')),
            'search' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|max:255',
            'phone' => 'sometimes|string|max:50',
            'created_by' => 'sometimes|integer|exists:users,id',
            'assigned_to_user_id' => 'sometimes|integer|exists:users,id',

            // Date filters
            'created_from' => 'sometimes|date',
            'created_to' => 'sometimes|date',
            'updated_from' => 'sometimes|date',
            'updated_to' => 'sometimes|date',

            // Pagination & sorting
            'page' => 'sometimes|integer|min:1',
            'per_page' => 'sometimes|integer|min:1|max:100',
            'order_by' => 'sometimes|string|in:id,created_at,updated_at,first_name,last_name,email',
            'order_direction' => 'sometimes|string|in:asc,desc',

            // Soft deletes
            'with_trashed' => 'sometimes|boolean',
            'only_trashed' => 'sometimes|boolean',

            // Relationships
            'with_lead_form' => 'sometimes|boolean',
            'with_form_fields' => 'sometimes|boolean',
            'with_users' => 'sometimes|boolean',
            'with_author' => 'sometimes|boolean',
            'with_meta' => 'sometimes|boolean',
        ]);

        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        return new self($data);
    }

    public function toArray(): array
    {
        return [
            'lead_form_id' => $this->lead_form_id,
            'lead_source' => $this->lead_source,
            'lead_statuses' => $this->lead_statuses ? array_map(fn($status) => $status->value, $this->lead_statuses) : null,
            'search' => $this->search,
            'email' => $this->email,
            'phone' => $this->phone,
            'created_by' => $this->created_by,
            'assigned_to_user_id' => $this->assigned_to_user_id,
            'created_from' => $this->created_from,
            'created_to' => $this->created_to,
            'updated_from' => $this->updated_from,
            'updated_to' => $this->updated_to,
            'page' => $this->page,
            'per_page' => $this->per_page,
            'order_by' => $this->order_by,
            'order_direction' => $this->order_direction,
            'with_trashed' => $this->with_trashed,
            'only_trashed' => $this->only_trashed,
        ];
    }

    public function getRelationsToLoad(): array
    {
        $relations = [];

        if ($this->with_lead_form) {
            $relations[] = 'leadForm';
        }
        if ($this->with_form_fields) {
            $relations[] = 'formFields';
        }
        if ($this->with_users) {
            $relations[] = 'users';
        }
        if ($this->with_author) {
            $relations[] = 'author';
        }

        return $relations;
    }
}
