<?php

namespace App\DTOs\Lead\LeadAddress;

use Illuminate\Http\Request;

class CreateLeadAddressDTO
{
    public int $lead_id;

    public ?string $address_line_1;

    public ?string $address_line_2;

    public ?int $city_id;

    public ?int $state_id;

    public ?int $country_id;

    public ?string $zip_code;

    public ?float $latitude;

    public ?float $longitude;

    // Optional location name strings (to create relations if IDs not provided)
    public ?string $city_name;

    public ?string $state_name;

    public ?string $state_code;

    public ?string $country_name;

    public ?string $country_code;

    public function __construct(array $data)
    {
        $this->lead_id = $data['lead_id'];
        $this->address_line_1 = $data['address_line_1'] ?? null;
        $this->address_line_2 = $data['address_line_2'] ?? null;
        $this->city_id = $data['city_id'] ?? null;
        $this->state_id = $data['state_id'] ?? null;
        $this->country_id = $data['country_id'] ?? null;
        $this->zip_code = $data['zip_code'] ?? null;
        $this->latitude = $data['latitude'] ?? null;
        $this->longitude = $data['longitude'] ?? null;

        // Location names for auto-creation
        $this->city_name = $data['city_name'] ?? null;
        $this->state_name = $data['state_name'] ?? null;
        $this->state_code = $data['state_code'] ?? null;
        $this->country_name = $data['country_name'] ?? null;
        $this->country_code = $data['country_code'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate([
            'lead_id' => 'required|integer|exists:leads,id',
            'address_line_1' => 'nullable|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city_id' => 'nullable|integer|exists:cities,id',
            'state_id' => 'nullable|integer|exists:states,id',
            'country_id' => 'nullable|integer|exists:countries,id',
            'zip_code' => 'nullable|string|max:20',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',

            // Location names
            'city_name' => 'nullable|string|max:255',
            'state_name' => 'nullable|string|max:255',
            'state_code' => 'nullable|string|max:10',
            'country_name' => 'nullable|string|max:255',
            'country_code' => 'nullable|string|size:2',
        ]);

        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        return new self($data);
    }

    /**
     * Create from Google Maps API response or similar geocoding data
     */
    public static function fromGeocodingData(int $leadId, array $geocodeData): self
    {
        return new self([
            'lead_id' => $leadId,
            'address_line_1' => $geocodeData['address_line_1'] ?? $geocodeData['formatted_address'] ?? null,
            'city_name' => $geocodeData['city'] ?? $geocodeData['locality'] ?? null,
            'state_name' => $geocodeData['state'] ?? $geocodeData['administrative_area_level_1'] ?? null,
            'country_name' => $geocodeData['country'] ?? null,
            'country_code' => $geocodeData['country_code'] ?? null,
            'zip_code' => $geocodeData['zip_code'] ?? $geocodeData['postal_code'] ?? null,
            'latitude' => $geocodeData['latitude'] ?? $geocodeData['lat'] ?? null,
            'longitude' => $geocodeData['longitude'] ?? $geocodeData['lng'] ?? null,
        ]);
    }

    public function toArray(): array
    {
        return [
            'lead_id' => $this->lead_id,
            'address_line_1' => $this->address_line_1,
            'address_line_2' => $this->address_line_2,
            'city_id' => $this->city_id,
            'state_id' => $this->state_id,
            'country_id' => $this->country_id,
            'zip_code' => $this->zip_code,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ];
    }

    public function hasLocationNames(): bool
    {
        return $this->city_name !== null
            || $this->state_name !== null
            || $this->country_name !== null;
    }

    public function getLocationNames(): array
    {
        return [
            'city_name' => $this->city_name,
            'state_name' => $this->state_name,
            'state_code' => $this->state_code,
            'country_name' => $this->country_name,
            'country_code' => $this->country_code,
        ];
    }

    public function hasCoordinates(): bool
    {
        return $this->latitude !== null && $this->longitude !== null;
    }
}
