<?php

namespace App\DTOs\Lead\LeadAddress;

use App\Models\Lead\LeadAddress;

class LeadAddressDTO
{
    public int $id;
    public int $lead_id;
    public ?string $address_line_1;
    public ?string $address_line_2;
    public ?int $city_id;
    public ?int $state_id;
    public ?int $country_id;
    public ?string $zip_code;
    public ?float $latitude;
    public ?float $longitude;

    // Relationships
    public ?array $lead;
    public ?array $city;
    public ?array $state;
    public ?array $country;

    public function __construct(LeadAddress $address, bool $includeRelations = false)
    {
        $this->id = $address->id;
        $this->lead_id = $address->lead_id;
        $this->address_line_1 = $address->address_line_1;
        $this->address_line_2 = $address->address_line_2;
        $this->city_id = $address->city_id;
        $this->state_id = $address->state_id;
        $this->country_id = $address->country_id;
        $this->zip_code = $address->zip_code;
        $this->latitude = $address->latitude;
        $this->longitude = $address->longitude;

        if ($includeRelations) {
            $this->lead = $address->lead ? [
                'id' => $address->lead->id,
                'uuid' => $address->lead->uuid,
                'first_name' => $address->lead->first_name,
                'last_name' => $address->lead->last_name,
            ] : null;

            $this->city = $address->city ? [
                'id' => $address->city->id,
                'city_name' => $address->city->city_name,
            ] : null;

            $this->state = $address->state ? [
                'id' => $address->state->id,
                'state_name' => $address->state->state_name,
                'state_code' => $address->state->state_code,
            ] : null;

            $this->country = $address->country ? [
                'id' => $address->country->id,
                'country_name' => $address->country->country_name,
                'country_code' => $address->country->country_code,
            ] : null;
        }
    }

    public function toArray(): array
    {
        $data = [
            'id' => $this->id,
            'lead_id' => $this->lead_id,
            'address_line_1' => $this->address_line_1,
            'address_line_2' => $this->address_line_2,
            'city_id' => $this->city_id,
            'state_id' => $this->state_id,
            'country_id' => $this->country_id,
            'zip_code' => $this->zip_code,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ];

        if (isset($this->lead)) {
            $data['lead'] = $this->lead;
        }
        if (isset($this->city)) {
            $data['city'] = $this->city;
        }
        if (isset($this->state)) {
            $data['state'] = $this->state;
        }
        if (isset($this->country)) {
            $data['country'] = $this->country;
        }

        return $data;
    }

    public function getFullAddress(): string
    {
        $parts = array_filter([
            $this->address_line_1,
            $this->address_line_2,
            $this->city['city_name'] ?? null,
            $this->state['state_name'] ?? null,
            $this->country['country_name'] ?? null,
            $this->zip_code,
        ]);

        return implode(', ', $parts);
    }

    public function hasCoordinates(): bool
    {
        return $this->latitude !== null && $this->longitude !== null;
    }

    public function getCoordinates(): ?array
    {
        if (!$this->hasCoordinates()) {
            return null;
        }

        return [
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ];
    }
}
