<?php

namespace App\DTOs\Lead\LeadAddress;

use Illuminate\Http\Request;

class UpdateLeadAddressDTO
{
    public ?string $address_line_1;
    public ?string $address_line_2;
    public ?int $city_id;
    public ?int $state_id;
    public ?int $country_id;
    public ?string $zip_code;
    public ?float $latitude;
    public ?float $longitude;

    // Optional location name strings
    public ?string $city_name;
    public ?string $state_name;
    public ?string $country_name;
    public ?string $country_code;

    public function __construct(array $data)
    {
        $this->address_line_1 = $data['address_line_1'] ?? null;
        $this->address_line_2 = $data['address_line_2'] ?? null;
        $this->city_id = $data['city_id'] ?? null;
        $this->state_id = $data['state_id'] ?? null;
        $this->country_id = $data['country_id'] ?? null;
        $this->zip_code = $data['zip_code'] ?? null;
        $this->latitude = $data['latitude'] ?? null;
        $this->longitude = $data['longitude'] ?? null;

        // Location names for auto-creation
        $this->city_name = $data['city_name'] ?? null;
        $this->state_name = $data['state_name'] ?? null;
        $this->country_name = $data['country_name'] ?? null;
        $this->country_code = $data['country_code'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate([
            'address_line_1' => 'sometimes|nullable|string|max:255',
            'address_line_2' => 'sometimes|nullable|string|max:255',
            'city_id' => 'sometimes|nullable|integer|exists:cities,id',
            'state_id' => 'sometimes|nullable|integer|exists:states,id',
            'country_id' => 'sometimes|nullable|integer|exists:countries,id',
            'zip_code' => 'sometimes|nullable|string|max:20',
            'latitude' => 'sometimes|nullable|numeric|between:-90,90',
            'longitude' => 'sometimes|nullable|numeric|between:-180,180',

            // Location names
            'city_name' => 'sometimes|nullable|string|max:255',
            'state_name' => 'sometimes|nullable|string|max:255',
            'country_name' => 'sometimes|nullable|string|max:255',
            'country_code' => 'sometimes|nullable|string|size:2',
        ]);

        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        return new self($data);
    }

    public function toArray(): array
    {
        $data = [];

        if ($this->address_line_1 !== null) {
            $data['address_line_1'] = $this->address_line_1;
        }
        if ($this->address_line_2 !== null) {
            $data['address_line_2'] = $this->address_line_2;
        }
        if ($this->city_id !== null) {
            $data['city_id'] = $this->city_id;
        }
        if ($this->state_id !== null) {
            $data['state_id'] = $this->state_id;
        }
        if ($this->country_id !== null) {
            $data['country_id'] = $this->country_id;
        }
        if ($this->zip_code !== null) {
            $data['zip_code'] = $this->zip_code;
        }
        if ($this->latitude !== null) {
            $data['latitude'] = $this->latitude;
        }
        if ($this->longitude !== null) {
            $data['longitude'] = $this->longitude;
        }

        return $data;
    }

    public function hasChanges(): bool
    {
        return !empty($this->toArray()) || $this->hasLocationNames();
    }

    public function hasLocationNames(): bool
    {
        return $this->city_name !== null
            || $this->state_name !== null
            || $this->country_name !== null;
    }

    public function getLocationNames(): array
    {
        return [
            'city_name' => $this->city_name,
            'state_name' => $this->state_name,
            'country_name' => $this->country_name,
            'country_code' => $this->country_code,
        ];
    }

    public function hasCoordinates(): bool
    {
        return $this->latitude !== null && $this->longitude !== null;
    }
}
