<?php

namespace App\DTOs\Lead\LeadForm;

use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class CreateLeadFormDTO
{
    public string $title;
    public ?string $description;
    public ?LeadFormMatchCriteria $match_criteria;
    public ?LeadFormReceptionMode $reception_mode;
    public bool $enable_captcha;
    public bool $is_active;
    public ?string $submit_message;
    public ?int $created_by;

    // Field attachments with pivot data
    public ?array $field_ids; // Array of field IDs
    public ?array $field_configs; // ['field_id' => ['placeholder' => '', 'is_required' => true, ...]]

    // Member categories with credits/radius
    public ?array $member_category_ids;
    public ?array $member_category_configs; // ['category_id' => ['credits' => 10, 'radius' => 50]]

    public function __construct(array $data)
    {
        $this->title = $data['title'];
        $this->description = $data['description'] ?? null;
        $this->match_criteria = isset($data['match_criteria'])
            ? LeadFormMatchCriteria::from($data['match_criteria'])
            : null;
        $this->reception_mode = isset($data['reception_mode'])
            ? LeadFormReceptionMode::from($data['reception_mode'])
            : null;
        $this->enable_captcha = $data['enable_captcha'] ?? false;
        $this->is_active = $data['is_active'] ?? true;
        $this->submit_message = $data['submit_message'] ?? null;
        $this->created_by = $data['created_by'] ?? null;
        $this->field_ids = $data['field_ids'] ?? null;
        $this->field_configs = $data['field_configs'] ?? null;
        $this->member_category_ids = $data['member_category_ids'] ?? null;
        $this->member_category_configs = $data['member_category_configs'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'match_criteria' => [
                'nullable',
                'string',
                Rule::in(array_column(LeadFormMatchCriteria::cases(), 'value'))
            ],
            'reception_mode' => [
                'nullable',
                'string',
                Rule::in(array_column(LeadFormReceptionMode::cases(), 'value'))
            ],
            'enable_captcha' => 'sometimes|boolean',
            'is_active' => 'sometimes|boolean',
            'submit_message' => 'nullable|string|max:500',
            'created_by' => 'nullable|integer|exists:users,id',

            // Field attachments
            'field_ids' => 'nullable|array',
            'field_ids.*' => 'integer|exists:lead_form_fields,id',
            'field_configs' => 'nullable|array',
            'field_configs.*.field_placeholder' => 'nullable|string|max:255',
            'field_configs.*.toggle_yes_text' => 'nullable|string|max:50',
            'field_configs.*.toggle_no_text' => 'nullable|string|max:50',
            'field_configs.*.field_columns' => 'nullable|integer|min:1|max:12',
            'field_configs.*.help_text' => 'nullable|string|max:500',
            'field_configs.*.field_options' => 'nullable|array',
            'field_configs.*.is_required' => 'nullable|boolean',
            'field_configs.*.is_visible' => 'nullable|boolean',
            'field_configs.*.field_order' => 'nullable|integer|min:0',

            // Member categories
            'member_category_ids' => 'nullable|array',
            'member_category_ids.*' => 'integer|exists:member_categories,id',
            'member_category_configs' => 'nullable|array',
            'member_category_configs.*.credits' => 'nullable|numeric|min:0',
            'member_category_configs.*.radius' => 'nullable|numeric|min:0',
        ];
    }

    public function toArray(): array
    {
        $data = [
            'title' => $this->title,
            'enable_captcha' => $this->enable_captcha,
            'is_active' => $this->is_active,
        ];

        if ($this->description !== null) {
            $data['description'] = $this->description;
        }
        if ($this->match_criteria !== null) {
            $data['match_criteria'] = $this->match_criteria->value;
        }
        if ($this->reception_mode !== null) {
            $data['reception_mode'] = $this->reception_mode->value;
        }
        if ($this->submit_message !== null) {
            $data['submit_message'] = $this->submit_message;
        }
        if ($this->created_by !== null) {
            $data['created_by'] = $this->created_by;
        }

        return $data;
    }

    public function getFieldAttachments(): ?array
    {
        if ($this->field_ids === null) {
            return null;
        }

        $attachments = [];
        foreach ($this->field_ids as $fieldId) {
            $config = $this->field_configs[$fieldId] ?? [];
            $attachments[$fieldId] = $config;
        }

        return $attachments;
    }

    public function getMemberCategoryAttachments(): ?array
    {
        if ($this->member_category_ids === null) {
            return null;
        }

        $attachments = [];
        foreach ($this->member_category_ids as $categoryId) {
            $config = $this->member_category_configs[$categoryId] ?? [];
            $attachments[$categoryId] = $config;
        }

        return $attachments;
    }
}
