<?php

namespace App\DTOs\Lead\LeadForm;

use App\Models\Lead\LeadForm;
use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;

class LeadFormDTO
{
    public int $id;
    public string $uuid;
    public string $title;
    public ?string $description;
    public ?LeadFormMatchCriteria $match_criteria;
    public ?LeadFormReceptionMode $reception_mode;
    public bool $enable_captcha;
    public bool $is_active;
    public ?string $submit_message;
    public ?int $created_by;
    public string $created_at;
    public string $updated_at;
    public ?string $deleted_at;

    // Relationships
    public ?array $visits;
    public ?int $visits_count;
    public ?array $leads;
    public ?int $leads_count;
    public ?array $form_fields;
    public ?array $author;
    public ?array $member_categories;

    public function __construct(LeadForm $leadForm, bool $includeRelations = false)
    {
        $this->id = $leadForm->id;
        $this->uuid = $leadForm->uuid;
        $this->title = $leadForm->title;
        $this->description = $leadForm->description;
        $this->match_criteria = $leadForm->match_criteria;
        $this->reception_mode = $leadForm->reception_mode;
        $this->enable_captcha = $leadForm->enable_captcha ?? false;
        $this->is_active = $leadForm->is_active ?? true;
        $this->submit_message = $leadForm->submit_message;
        $this->created_by = $leadForm->created_by;
        $this->created_at = $leadForm->created_at->toIso8601String();
        $this->updated_at = $leadForm->updated_at->toIso8601String();
        $this->deleted_at = $leadForm->deleted_at?->toIso8601String();

        if ($includeRelations) {
            $this->visits = $leadForm->visits?->map(fn($visit) => [
                'id' => $visit->id,
                'source' => $visit->source,
                'visit_date' => $visit->visit_date,
            ])->toArray();

            $this->visits_count = $leadForm->visits_count ?? $leadForm->visits?->count();

            $this->leads = $leadForm->leads?->map(fn($lead) => [
                'id' => $lead->id,
                'uuid' => $lead->uuid,
                'first_name' => $lead->first_name,
                'last_name' => $lead->last_name,
                'email' => $lead->email,
                'lead_status' => $lead->lead_status?->value,
            ])->toArray();

            $this->leads_count = $leadForm->leads_count ?? $leadForm->leads?->count();

            $this->form_fields = $leadForm->formFields?->map(fn($field) => [
                'id' => $field->id,
                'field_name' => $field->field_name,
                'field_type' => $field->field_type?->value,
                'data_type' => $field->data_type?->value,
                'is_required' => $field->pivot->is_required ?? false,
                'is_visible' => $field->pivot->is_visible ?? true,
                'field_order' => $field->pivot->field_order ?? 0,
                'field_placeholder' => $field->pivot->field_placeholder,
                'help_text' => $field->pivot->help_text,
            ])->toArray();

            $this->author = $leadForm->author ? [
                'id' => $leadForm->author->id,
                'name' => $leadForm->author->name,
                'email' => $leadForm->author->email,
            ] : null;

            $this->member_categories = $leadForm->memberCategories?->map(fn($category) => [
                'id' => $category->id,
                'name' => $category->name,
                'credits' => $category->pivot->credits ?? null,
                'radius' => $category->pivot->radius ?? null,
            ])->toArray();
        }
    }

    public function toArray(): array
    {
        $data = [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'title' => $this->title,
            'description' => $this->description,
            'match_criteria' => $this->match_criteria?->value,
            'reception_mode' => $this->reception_mode?->value,
            'enable_captcha' => $this->enable_captcha,
            'is_active' => $this->is_active,
            'submit_message' => $this->submit_message,
            'created_by' => $this->created_by,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            'deleted_at' => $this->deleted_at,
        ];

        if (isset($this->visits)) {
            $data['visits'] = $this->visits;
        }
        if (isset($this->visits_count)) {
            $data['visits_count'] = $this->visits_count;
        }
        if (isset($this->leads)) {
            $data['leads'] = $this->leads;
        }
        if (isset($this->leads_count)) {
            $data['leads_count'] = $this->leads_count;
        }
        if (isset($this->form_fields)) {
            $data['form_fields'] = $this->form_fields;
        }
        if (isset($this->author)) {
            $data['author'] = $this->author;
        }
        if (isset($this->member_categories)) {
            $data['member_categories'] = $this->member_categories;
        }

        return $data;
    }
}
