<?php

namespace App\DTOs\Lead\LeadForm;

use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class LeadFormFilterDTO
{
    public ?string $search; // Search in title, description
    public ?bool $is_active;
    public ?LeadFormMatchCriteria $match_criteria;
    public ?LeadFormReceptionMode $reception_mode;
    public ?bool $enable_captcha;
    public ?int $created_by;
    public ?int $member_category_id;

    // Date filtering
    public ?string $created_from;
    public ?string $created_to;
    public ?string $updated_from;
    public ?string $updated_to;

    // Stats filtering
    public ?int $min_visits;
    public ?int $min_leads;

    // Pagination & sorting
    public int $page;
    public int $per_page;
    public string $order_by;
    public string $order_direction;

    // Include soft deleted
    public bool $with_trashed;
    public bool $only_trashed;

    // Include relationships
    public bool $with_visits;
    public bool $with_leads;
    public bool $with_form_fields;
    public bool $with_author;
    public bool $with_member_categories;
    public bool $with_counts; // Load visits_count and leads_count

    public function __construct(array $data)
    {
        $this->search = $data['search'] ?? null;
        $this->is_active = $data['is_active'] ?? null;
        $this->match_criteria = isset($data['match_criteria'])
            ? LeadFormMatchCriteria::from($data['match_criteria'])
            : null;
        $this->reception_mode = isset($data['reception_mode'])
            ? LeadFormReceptionMode::from($data['reception_mode'])
            : null;
        $this->enable_captcha = $data['enable_captcha'] ?? null;
        $this->created_by = $data['created_by'] ?? null;
        $this->member_category_id = $data['member_category_id'] ?? null;

        // Date filters
        $this->created_from = $data['created_from'] ?? null;
        $this->created_to = $data['created_to'] ?? null;
        $this->updated_from = $data['updated_from'] ?? null;
        $this->updated_to = $data['updated_to'] ?? null;

        // Stats filters
        $this->min_visits = $data['min_visits'] ?? null;
        $this->min_leads = $data['min_leads'] ?? null;

        // Pagination & sorting
        $this->page = $data['page'] ?? 1;
        $this->per_page = $data['per_page'] ?? 15;
        $this->order_by = $data['order_by'] ?? 'created_at';
        $this->order_direction = $data['order_direction'] ?? 'desc';

        // Soft deletes
        $this->with_trashed = $data['with_trashed'] ?? false;
        $this->only_trashed = $data['only_trashed'] ?? false;

        // Relationships
        $this->with_visits = $data['with_visits'] ?? false;
        $this->with_leads = $data['with_leads'] ?? false;
        $this->with_form_fields = $data['with_form_fields'] ?? false;
        $this->with_author = $data['with_author'] ?? false;
        $this->with_member_categories = $data['with_member_categories'] ?? false;
        $this->with_counts = $data['with_counts'] ?? false;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate([
            'search' => 'sometimes|string|max:255',
            'is_active' => 'sometimes|boolean',
            'match_criteria' => [
                'sometimes',
                'string',
                Rule::in(array_column(LeadFormMatchCriteria::cases(), 'value'))
            ],
            'reception_mode' => [
                'sometimes',
                'string',
                Rule::in(array_column(LeadFormReceptionMode::cases(), 'value'))
            ],
            'enable_captcha' => 'sometimes|boolean',
            'created_by' => 'sometimes|integer|exists:users,id',
            'member_category_id' => 'sometimes|integer|exists:member_categories,id',

            // Date filters
            'created_from' => 'sometimes|date',
            'created_to' => 'sometimes|date',
            'updated_from' => 'sometimes|date',
            'updated_to' => 'sometimes|date',

            // Stats filters
            'min_visits' => 'sometimes|integer|min:0',
            'min_leads' => 'sometimes|integer|min:0',

            // Pagination & sorting
            'page' => 'sometimes|integer|min:1',
            'per_page' => 'sometimes|integer|min:1|max:100',
            'order_by' => 'sometimes|string|in:id,title,created_at,updated_at,is_active',
            'order_direction' => 'sometimes|string|in:asc,desc',

            // Soft deletes
            'with_trashed' => 'sometimes|boolean',
            'only_trashed' => 'sometimes|boolean',

            // Relationships
            'with_visits' => 'sometimes|boolean',
            'with_leads' => 'sometimes|boolean',
            'with_form_fields' => 'sometimes|boolean',
            'with_author' => 'sometimes|boolean',
            'with_member_categories' => 'sometimes|boolean',
            'with_counts' => 'sometimes|boolean',
        ]);

        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        return new self($data);
    }

    public function toArray(): array
    {
        return [
            'search' => $this->search,
            'is_active' => $this->is_active,
            'match_criteria' => $this->match_criteria?->value,
            'reception_mode' => $this->reception_mode?->value,
            'enable_captcha' => $this->enable_captcha,
            'created_by' => $this->created_by,
            'member_category_id' => $this->member_category_id,
            'created_from' => $this->created_from,
            'created_to' => $this->created_to,
            'updated_from' => $this->updated_from,
            'updated_to' => $this->updated_to,
            'min_visits' => $this->min_visits,
            'min_leads' => $this->min_leads,
            'page' => $this->page,
            'per_page' => $this->per_page,
            'order_by' => $this->order_by,
            'order_direction' => $this->order_direction,
            'with_trashed' => $this->with_trashed,
            'only_trashed' => $this->only_trashed,
        ];
    }

    public function getRelationsToLoad(): array
    {
        $relations = [];

        if ($this->with_visits) {
            $relations[] = 'visits';
        }
        if ($this->with_leads) {
            $relations[] = 'leads';
        }
        if ($this->with_form_fields) {
            $relations[] = 'formFields';
        }
        if ($this->with_author) {
            $relations[] = 'author';
        }
        if ($this->with_member_categories) {
            $relations[] = 'memberCategories';
        }

        return $relations;
    }

    public function getCountsToLoad(): array
    {
        if (!$this->with_counts) {
            return [];
        }

        return ['visits', 'leads'];
    }
}
