<?php

namespace App\DTOs\Lead\LeadForm;

use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class UpdateLeadFormDTO
{
    public ?string $title;
    public ?string $description;
    public ?LeadFormMatchCriteria $match_criteria;
    public ?LeadFormReceptionMode $reception_mode;
    public ?bool $enable_captcha;
    public ?bool $is_active;
    public ?string $submit_message;

    // Field updates
    public ?array $field_ids;
    public ?array $field_configs;
    public ?bool $sync_fields; // Whether to sync or keep existing

    // Member category updates
    public ?array $member_category_ids;
    public ?array $member_category_configs;
    public ?bool $sync_member_categories;

    public function __construct(array $data)
    {
        $this->title = $data['title'] ?? null;
        $this->description = $data['description'] ?? null;
        $this->match_criteria = isset($data['match_criteria'])
            ? LeadFormMatchCriteria::from($data['match_criteria'])
            : null;
        $this->reception_mode = isset($data['reception_mode'])
            ? LeadFormReceptionMode::from($data['reception_mode'])
            : null;
        $this->enable_captcha = $data['enable_captcha'] ?? null;
        $this->is_active = $data['is_active'] ?? null;
        $this->submit_message = $data['submit_message'] ?? null;
        $this->field_ids = $data['field_ids'] ?? null;
        $this->field_configs = $data['field_configs'] ?? null;
        $this->sync_fields = $data['sync_fields'] ?? true;
        $this->member_category_ids = $data['member_category_ids'] ?? null;
        $this->member_category_configs = $data['member_category_configs'] ?? null;
        $this->sync_member_categories = $data['sync_member_categories'] ?? true;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|nullable|string|max:1000',
            'match_criteria' => [
                'sometimes',
                'nullable',
                'string',
                Rule::in(array_column(LeadFormMatchCriteria::cases(), 'value'))
            ],
            'reception_mode' => [
                'sometimes',
                'nullable',
                'string',
                Rule::in(array_column(LeadFormReceptionMode::cases(), 'value'))
            ],
            'enable_captcha' => 'sometimes|boolean',
            'is_active' => 'sometimes|boolean',
            'submit_message' => 'sometimes|nullable|string|max:500',

            // Field updates
            'field_ids' => 'sometimes|array',
            'field_ids.*' => 'integer|exists:lead_form_fields,id',
            'field_configs' => 'sometimes|array',
            'field_configs.*.field_placeholder' => 'nullable|string|max:255',
            'field_configs.*.toggle_yes_text' => 'nullable|string|max:50',
            'field_configs.*.toggle_no_text' => 'nullable|string|max:50',
            'field_configs.*.field_columns' => 'nullable|integer|min:1|max:12',
            'field_configs.*.help_text' => 'nullable|string|max:500',
            'field_configs.*.field_options' => 'nullable|array',
            'field_configs.*.is_required' => 'nullable|boolean',
            'field_configs.*.is_visible' => 'nullable|boolean',
            'field_configs.*.field_order' => 'nullable|integer|min:0',
            'sync_fields' => 'sometimes|boolean',

            // Member categories
            'member_category_ids' => 'sometimes|array',
            'member_category_ids.*' => 'integer|exists:member_categories,id',
            'member_category_configs' => 'sometimes|array',
            'member_category_configs.*.credits' => 'nullable|numeric|min:0',
            'member_category_configs.*.radius' => 'nullable|numeric|min:0',
            'sync_member_categories' => 'sometimes|boolean',
        ];
    }

    public function toArray(): array
    {
        $data = [];

        if ($this->title !== null) {
            $data['title'] = $this->title;
        }
        if ($this->description !== null) {
            $data['description'] = $this->description;
        }
        if ($this->match_criteria !== null) {
            $data['match_criteria'] = $this->match_criteria->value;
        }
        if ($this->reception_mode !== null) {
            $data['reception_mode'] = $this->reception_mode->value;
        }
        if ($this->enable_captcha !== null) {
            $data['enable_captcha'] = $this->enable_captcha;
        }
        if ($this->is_active !== null) {
            $data['is_active'] = $this->is_active;
        }
        if ($this->submit_message !== null) {
            $data['submit_message'] = $this->submit_message;
        }

        return $data;
    }

    public function hasChanges(): bool
    {
        return !empty($this->toArray())
            || $this->field_ids !== null
            || $this->member_category_ids !== null;
    }

    public function getFieldAttachments(): ?array
    {
        if ($this->field_ids === null) {
            return null;
        }

        $attachments = [];
        foreach ($this->field_ids as $fieldId) {
            $config = $this->field_configs[$fieldId] ?? [];
            $attachments[$fieldId] = $config;
        }

        return $attachments;
    }

    public function shouldSyncFields(): bool
    {
        return $this->sync_fields ?? true;
    }

    public function getMemberCategoryAttachments(): ?array
    {
        if ($this->member_category_ids === null) {
            return null;
        }

        $attachments = [];
        foreach ($this->member_category_ids as $categoryId) {
            $config = $this->member_category_configs[$categoryId] ?? [];
            $attachments[$categoryId] = $config;
        }

        return $attachments;
    }

    public function shouldSyncMemberCategories(): bool
    {
        return $this->sync_member_categories ?? true;
    }
}
