<?php

namespace App\DTOs\Lead\LeadFormFields;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class AttachFieldToFormDTO
{
    public int $lead_form_id;
    public int $field_id;
    public ?string $field_placeholder;
    public ?string $toggle_yes_text;
    public ?string $toggle_no_text;
    public ?int $field_columns;
    public ?string $help_text;
    public ?array $field_options;
    public bool $is_required;
    public bool $is_visible;
    public int $field_order;

    public function __construct(array $data)
    {
        $this->lead_form_id = $data['lead_form_id'];
        $this->field_id = $data['field_id'];
        $this->field_placeholder = $data['field_placeholder'] ?? null;
        $this->toggle_yes_text = $data['toggle_yes_text'] ?? null;
        $this->toggle_no_text = $data['toggle_no_text'] ?? null;
        $this->field_columns = $data['field_columns'] ?? 12;
        $this->help_text = $data['help_text'] ?? null;
        $this->field_options = $data['field_options'] ?? null;
        $this->is_required = $data['is_required'] ?? false;
        $this->is_visible = $data['is_visible'] ?? true;
        $this->field_order = $data['field_order'] ?? 0;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'lead_form_id' => 'required|integer|exists:lead_forms,id',
            'field_id' => 'required|integer|exists:lead_form_fields,id',
            'field_placeholder' => 'nullable|string|max:255',
            'toggle_yes_text' => 'nullable|string|max:50',
            'toggle_no_text' => 'nullable|string|max:50',
            'field_columns' => 'nullable|integer|min:1|max:12',
            'help_text' => 'nullable|string|max:500',
            'field_options' => 'nullable|array',
            'field_options.*' => 'string|max:255',
            'is_required' => 'sometimes|boolean',
            'is_visible' => 'sometimes|boolean',
            'field_order' => 'sometimes|integer|min:0',
        ];
    }

    public function toArray(): array
    {
        return [
            'lead_form_id' => $this->lead_form_id,
            'field_id' => $this->field_id,
            'field_placeholder' => $this->field_placeholder,
            'toggle_yes_text' => $this->toggle_yes_text,
            'toggle_no_text' => $this->toggle_no_text,
            'field_columns' => $this->field_columns,
            'help_text' => $this->help_text,
            'field_options' => $this->field_options,
            'is_required' => $this->is_required,
            'is_visible' => $this->is_visible,
            'field_order' => $this->field_order,
        ];
    }

    public function getPivotData(): array
    {
        return [
            'field_placeholder' => $this->field_placeholder,
            'toggle_yes_text' => $this->toggle_yes_text,
            'toggle_no_text' => $this->toggle_no_text,
            'field_columns' => $this->field_columns,
            'help_text' => $this->help_text,
            'field_options' => $this->field_options,
            'is_required' => $this->is_required,
            'is_visible' => $this->is_visible,
            'field_order' => $this->field_order,
        ];
    }
}
