<?php

namespace App\DTOs\Lead\LeadFormFields;

use App\Enums\LeadFormFieldType;
use App\Enums\LeadFormFieldDataType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class CreateLeadFormFieldDTO
{
    public string $field_name;
    public string $field_id;
    public ?LeadFormFieldType $field_type;
    public ?LeadFormFieldDataType $data_type;
    public bool $is_default;
    public ?int $created_by;

    public function __construct(array $data)
    {
        $this->field_name = $data['field_name'];
        $this->field_id = $data['field_id'];
        $this->field_type = isset($data['field_type'])
            ? LeadFormFieldType::from($data['field_type'])
            : null;
        $this->data_type = isset($data['data_type'])
            ? LeadFormFieldDataType::from($data['data_type'])
            : null;
        $this->is_default = $data['is_default'] ?? false;
        $this->created_by = $data['created_by'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'field_name' => 'required|string|max:255',
            'field_id' => 'required|string|max:255|unique:lead_form_fields,field_id',
            'field_type' => [
                'nullable',
                'string',
                Rule::in(array_column(LeadFormFieldType::cases(), 'value'))
            ],
            'data_type' => [
                'nullable',
                'string',
                Rule::in(array_column(LeadFormFieldDataType::cases(), 'value'))
            ],
            'is_default' => 'sometimes|boolean',
            'created_by' => 'nullable|integer|exists:users,id',
        ];
    }

    public function toArray(): array
    {
        $data = [
            'field_name' => $this->field_name,
            'field_id' => $this->field_id,
            'is_default' => $this->is_default,
        ];

        if ($this->field_type !== null) {
            $data['field_type'] = $this->field_type->value;
        }
        if ($this->data_type !== null) {
            $data['data_type'] = $this->data_type->value;
        }
        if ($this->created_by !== null) {
            $data['created_by'] = $this->created_by;
        }

        return $data;
    }
}
