<?php

namespace App\DTOs\Lead\LeadFormFields;

use App\Models\Lead\LeadFormFields;
use App\Enums\LeadFormFieldType;
use App\Enums\LeadFormFieldDataType;

class LeadFormFieldDTO
{
    public int $id;
    public string $uuid;
    public string $field_name;
    public string $field_id;
    public ?LeadFormFieldType $field_type;
    public ?LeadFormFieldDataType $data_type;
    public bool $is_default;
    public ?int $created_by;
    public string $created_at;
    public string $updated_at;
    public ?string $deleted_at;

    // Additional properties when loaded with pivot
    public ?string $field_placeholder;
    public ?string $toggle_yes_text;
    public ?string $toggle_no_text;
    public ?int $field_columns;
    public ?string $help_text;
    public ?array $field_options;
    public ?bool $is_required;
    public ?bool $is_visible;
    public ?int $field_order;

    // Relationships
    public ?array $author;

    public function __construct(LeadFormFields $field, bool $includeRelations = false)
    {
        $this->id = $field->id;
        $this->uuid = $field->uuid;
        $this->field_name = $field->field_name;
        $this->field_id = $field->field_id;
        $this->field_type = $field->field_type;
        $this->data_type = $field->data_type;
        $this->is_default = $field->is_default ?? false;
        $this->created_by = $field->created_by;
        $this->created_at = $field->created_at->toIso8601String();
        $this->updated_at = $field->updated_at->toIso8601String();
        $this->deleted_at = $field->deleted_at?->toIso8601String();

        // Load pivot data if available
        if (isset($field->pivot)) {
            $this->field_placeholder = $field->pivot->field_placeholder ?? null;
            $this->toggle_yes_text = $field->pivot->toggle_yes_text ?? null;
            $this->toggle_no_text = $field->pivot->toggle_no_text ?? null;
            $this->field_columns = $field->pivot->field_columns ?? null;
            $this->help_text = $field->pivot->help_text ?? null;
            $this->field_options = $field->pivot->field_options ?? null;
            $this->is_required = $field->pivot->is_required ?? null;
            $this->is_visible = $field->pivot->is_visible ?? null;
            $this->field_order = $field->pivot->field_order ?? null;
        }

        if ($includeRelations) {
            $this->author = $field->author ? [
                'id' => $field->author->id,
                'name' => $field->author->name,
                'email' => $field->author->email,
            ] : null;
        }
    }

    public function toArray(): array
    {
        $data = [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'field_name' => $this->field_name,
            'field_id' => $this->field_id,
            'field_type' => $this->field_type?->value,
            'data_type' => $this->data_type?->value,
            'is_default' => $this->is_default,
            'created_by' => $this->created_by,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            'deleted_at' => $this->deleted_at,
        ];

        // Include pivot data if available
        if ($this->field_placeholder !== null) {
            $data['field_placeholder'] = $this->field_placeholder;
        }
        if ($this->toggle_yes_text !== null) {
            $data['toggle_yes_text'] = $this->toggle_yes_text;
        }
        if ($this->toggle_no_text !== null) {
            $data['toggle_no_text'] = $this->toggle_no_text;
        }
        if ($this->field_columns !== null) {
            $data['field_columns'] = $this->field_columns;
        }
        if ($this->help_text !== null) {
            $data['help_text'] = $this->help_text;
        }
        if ($this->field_options !== null) {
            $data['field_options'] = $this->field_options;
        }
        if ($this->is_required !== null) {
            $data['is_required'] = $this->is_required;
        }
        if ($this->is_visible !== null) {
            $data['is_visible'] = $this->is_visible;
        }
        if ($this->field_order !== null) {
            $data['field_order'] = $this->field_order;
        }

        if (isset($this->author)) {
            $data['author'] = $this->author;
        }

        return $data;
    }

    public function isToggleField(): bool
    {
        return $this->field_type === LeadFormFieldType::Toggle;
    }

    public function hasOptions(): bool
    {
        return in_array($this->field_type, [
            LeadFormFieldType::Select,
            LeadFormFieldType::Radio,
            LeadFormFieldType::Checkbox,
        ]);
    }
}
