<?php

namespace App\DTOs\Lead\LeadFormFields;

use App\Enums\LeadFormFieldType;
use App\Enums\LeadFormFieldDataType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class UpdateLeadFormFieldDTO
{
    public ?string $field_name;
    public ?string $field_id;
    public ?LeadFormFieldType $field_type;
    public ?LeadFormFieldDataType $data_type;
    public ?bool $is_default;

    public function __construct(array $data)
    {
        $this->field_name = $data['field_name'] ?? null;
        $this->field_id = $data['field_id'] ?? null;
        $this->field_type = isset($data['field_type'])
            ? LeadFormFieldType::from($data['field_type'])
            : null;
        $this->data_type = isset($data['data_type'])
            ? LeadFormFieldDataType::from($data['data_type'])
            : null;
        $this->is_default = $data['is_default'] ?? null;
    }

    public static function fromRequest(Request $request, int $fieldId): self
    {
        $validated = $request->validate(self::rules($fieldId));
        return new self($validated);
    }

    public static function fromArray(array $data, ?int $fieldId = null): self
    {
        $validator = Validator::make($data, self::rules($fieldId));

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(?int $fieldId = null): array
    {
        return [
            'field_name' => 'sometimes|string|max:255',
            'field_id' => [
                'sometimes',
                'string',
                'max:255',
                $fieldId ? Rule::unique('lead_form_fields', 'field_id')->ignore($fieldId) : Rule::unique('lead_form_fields', 'field_id')
            ],
            'field_type' => [
                'sometimes',
                'nullable',
                'string',
                Rule::in(array_column(LeadFormFieldType::cases(), 'value'))
            ],
            'data_type' => [
                'sometimes',
                'nullable',
                'string',
                Rule::in(array_column(LeadFormFieldDataType::cases(), 'value'))
            ],
            'is_default' => 'sometimes|boolean',
        ];
    }

    public function toArray(): array
    {
        $data = [];

        if ($this->field_name !== null) {
            $data['field_name'] = $this->field_name;
        }
        if ($this->field_id !== null) {
            $data['field_id'] = $this->field_id;
        }
        if ($this->field_type !== null) {
            $data['field_type'] = $this->field_type->value;
        }
        if ($this->data_type !== null) {
            $data['data_type'] = $this->data_type->value;
        }
        if ($this->is_default !== null) {
            $data['is_default'] = $this->is_default;
        }

        return $data;
    }

    public function hasChanges(): bool
    {
        return !empty($this->toArray());
    }
}
