<?php

namespace App\DTOs\Lead\LeadFormVisit;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class CreateLeadFormVisitDTO
{
    public int $lead_form_id;
    public ?string $source;
    public ?string $unique_identifier;
    public ?string $ip_address;
    public ?string $country;
    public ?string $state;
    public ?string $city;
    public ?string $visit_date;

    // Additional geolocation data
    public ?float $latitude;
    public ?float $longitude;
    public ?string $user_agent;
    public ?string $referrer;

    public function __construct(array $data)
    {
        $this->lead_form_id = $data['lead_form_id'];
        $this->source = $data['source'] ?? null;
        $this->unique_identifier = $data['unique_identifier'] ?? null;
        $this->ip_address = $data['ip_address'] ?? null;
        $this->country = $data['country'] ?? null;
        $this->state = $data['state'] ?? null;
        $this->city = $data['city'] ?? null;
        $this->visit_date = $data['visit_date'] ?? now()->toDateTimeString();
        $this->latitude = $data['latitude'] ?? null;
        $this->longitude = $data['longitude'] ?? null;
        $this->user_agent = $data['user_agent'] ?? null;
        $this->referrer = $data['referrer'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate(self::rules());
        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        $validator = Validator::make($data, self::rules());

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        return new self($validator->validated());
    }

    /**
     * Get validation rules
     */
    protected static function rules(): array
    {
        return [
            'lead_form_id' => 'required|integer|exists:lead_forms,id',
            'source' => 'nullable|string|max:255',
            'unique_identifier' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
            'country' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'visit_date' => 'nullable|date',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'user_agent' => 'nullable|string|max:500',
            'referrer' => 'nullable|string|max:500',
        ];
    }

    /**
     * Automatically capture visit data from request
     */
    public static function autoCapture(Request $request, int $leadFormId): self
    {
        return new self([
            'lead_form_id' => $leadFormId,
            'source' => $request->query('source', $request->query('utm_source')),
            'unique_identifier' => $request->cookie('visitor_id') ?? session()->getId(),
            'ip_address' => $request->ip(),
            'visit_date' => now()->toDateTimeString(),
            'user_agent' => $request->userAgent(),
            'referrer' => $request->header('referer'),
        ]);
    }

    public function toArray(): array
    {
        return [
            'lead_form_id' => $this->lead_form_id,
            'source' => $this->source,
            'unique_identifier' => $this->unique_identifier,
            'ip_address' => $this->ip_address,
            'country' => $this->country,
            'state' => $this->state,
            'city' => $this->city,
            'visit_date' => $this->visit_date,
        ];
    }

    public function hasGeolocation(): bool
    {
        return $this->latitude !== null && $this->longitude !== null;
    }

    public function getGeolocation(): ?array
    {
        if (!$this->hasGeolocation()) {
            return null;
        }

        return [
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ];
    }

    public function getMetadata(): array
    {
        return array_filter([
            'user_agent' => $this->user_agent,
            'referrer' => $this->referrer,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ]);
    }
}
