<?php

namespace App\DTOs\Lead\LeadFormVisit;

use Illuminate\Http\Request;

class LeadFormVisitFilterDTO
{
    public ?int $lead_form_id;
    public ?string $source;
    public ?string $country;
    public ?string $state;
    public ?string $city;
    public ?string $ip_address;
    public ?string $unique_identifier;

    // Date filtering
    public ?string $visit_date_from;
    public ?string $visit_date_to;

    // Pagination & sorting
    public int $page;
    public int $per_page;
    public string $order_by;
    public string $order_direction;

    // Include relationships
    public bool $with_lead_form;

    // Grouping options
    public ?string $group_by; // source, country, state, city, visit_date

    public function __construct(array $data)
    {
        $this->lead_form_id = $data['lead_form_id'] ?? null;
        $this->source = $data['source'] ?? null;
        $this->country = $data['country'] ?? null;
        $this->state = $data['state'] ?? null;
        $this->city = $data['city'] ?? null;
        $this->ip_address = $data['ip_address'] ?? null;
        $this->unique_identifier = $data['unique_identifier'] ?? null;

        // Date filters
        $this->visit_date_from = $data['visit_date_from'] ?? null;
        $this->visit_date_to = $data['visit_date_to'] ?? null;

        // Pagination & sorting
        $this->page = $data['page'] ?? 1;
        $this->per_page = $data['per_page'] ?? 15;
        $this->order_by = $data['order_by'] ?? 'visit_date';
        $this->order_direction = $data['order_direction'] ?? 'desc';

        // Relationships
        $this->with_lead_form = $data['with_lead_form'] ?? false;

        // Grouping
        $this->group_by = $data['group_by'] ?? null;
    }

    public static function fromRequest(Request $request): self
    {
        $validated = $request->validate([
            'lead_form_id' => 'sometimes|integer|exists:lead_forms,id',
            'source' => 'sometimes|string|max:255',
            'country' => 'sometimes|string|max:255',
            'state' => 'sometimes|string|max:255',
            'city' => 'sometimes|string|max:255',
            'ip_address' => 'sometimes|ip',
            'unique_identifier' => 'sometimes|string|max:255',

            // Date filters
            'visit_date_from' => 'sometimes|date',
            'visit_date_to' => 'sometimes|date',

            // Pagination & sorting
            'page' => 'sometimes|integer|min:1',
            'per_page' => 'sometimes|integer|min:1|max:100',
            'order_by' => 'sometimes|string|in:id,visit_date,source,country,state,city',
            'order_direction' => 'sometimes|string|in:asc,desc',

            // Relationships
            'with_lead_form' => 'sometimes|boolean',

            // Grouping
            'group_by' => 'sometimes|string|in:source,country,state,city,visit_date',
        ]);

        return new self($validated);
    }

    public static function fromArray(array $data): self
    {
        return new self($data);
    }

    public function toArray(): array
    {
        return [
            'lead_form_id' => $this->lead_form_id,
            'source' => $this->source,
            'country' => $this->country,
            'state' => $this->state,
            'city' => $this->city,
            'ip_address' => $this->ip_address,
            'unique_identifier' => $this->unique_identifier,
            'visit_date_from' => $this->visit_date_from,
            'visit_date_to' => $this->visit_date_to,
            'page' => $this->page,
            'per_page' => $this->per_page,
            'order_by' => $this->order_by,
            'order_direction' => $this->order_direction,
            'group_by' => $this->group_by,
        ];
    }

    public function getRelationsToLoad(): array
    {
        $relations = [];

        if ($this->with_lead_form) {
            $relations[] = 'leadForm';
        }

        return $relations;
    }

    public function shouldGroup(): bool
    {
        return $this->group_by !== null;
    }
}
