<?php

namespace App\Enums;

enum FileUploadType: string
{
    case Image = 'image';
    case PDF = 'pdf';
    case Document = 'document';
    case Spreadsheet = 'spreadsheet';
    case Any = 'any';

    /**
     * Get the display label for the file type
     */
    public function label(): string
    {
        return match ($this) {
            self::Image => 'Images (JPG, PNG, GIF, WebP)',
            self::PDF => 'PDF Documents',
            self::Document => 'Documents (DOC, DOCX, ODT)',
            self::Spreadsheet => 'Spreadsheets (XLS, XLSX, CSV)',
            self::Any => 'Any File Type',
        };
    }

    /**
     * Get accepted MIME types for this file type
     */
    public function getMimeTypes(): array
    {
        return match ($this) {
            self::Image => [
                'image/jpeg',
                'image/png',
                'image/gif',
                'image/webp',
                'image/svg+xml',
            ],
            self::PDF => [
                'application/pdf',
            ],
            self::Document => [
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'application/vnd.oasis.opendocument.text',
                'text/plain',
            ],
            self::Spreadsheet => [
                'application/vnd.ms-excel',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'text/csv',
                'application/vnd.oasis.opendocument.spreadsheet',
            ],
            self::Any => ['*/*'],
        };
    }

    /**
     * Get file extensions for this file type
     */
    public function getExtensions(): array
    {
        return match ($this) {
            self::Image => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
            self::PDF => ['pdf'],
            self::Document => ['doc', 'docx', 'odt', 'txt'],
            self::Spreadsheet => ['xls', 'xlsx', 'csv', 'ods'],
            self::Any => ['*'],
        };
    }

    /**
     * Get the accept attribute value for HTML file input
     */
    public function getAcceptAttribute(): string
    {
        return match ($this) {
            self::Image => 'image/*',
            self::PDF => '.pdf,application/pdf',
            self::Document => '.doc,.docx,.odt,.txt',
            self::Spreadsheet => '.xls,.xlsx,.csv,.ods',
            self::Any => '*/*',
        };
    }

    /**
     * Get recommended max file size in KB
     */
    public function getRecommendedMaxSize(): int
    {
        return match ($this) {
            self::Image => 2048,      // 2MB
            self::PDF => 5120,        // 5MB
            self::Document => 5120,   // 5MB
            self::Spreadsheet => 5120, // 5MB
            self::Any => 10240,       // 10MB
        };
    }

    /**
     * Get all MIME types for multiple file type selections
     */
    public static function getCombinedMimeTypes(array $types): array
    {
        $mimeTypes = [];

        foreach ($types as $type) {
            if ($type instanceof self) {
                $mimeTypes = array_merge($mimeTypes, $type->getMimeTypes());
            } elseif (is_string($type)) {
                $enum = self::tryFrom($type);
                if ($enum) {
                    $mimeTypes = array_merge($mimeTypes, $enum->getMimeTypes());
                }
            }
        }

        return array_unique($mimeTypes);
    }

    /**
     * Get combined accept attribute for multiple file types
     */
    public static function getCombinedAcceptAttribute(array $types): string
    {
        $accepts = [];

        foreach ($types as $type) {
            if ($type instanceof self) {
                $accepts[] = $type->getAcceptAttribute();
            } elseif (is_string($type)) {
                $enum = self::tryFrom($type);
                if ($enum) {
                    $accepts[] = $enum->getAcceptAttribute();
                }
            }
        }

        return implode(',', array_unique($accepts));
    }

    /**
     * Get combined file extensions for multiple file types
     */
    public static function getCombinedExtensions(array $types): array
    {
        $extensions = [];

        foreach ($types as $type) {
            if ($type instanceof self) {
                $extensions = array_merge($extensions, $type->getExtensions());
            } elseif (is_string($type)) {
                $enum = self::tryFrom($type);
                if ($enum) {
                    $extensions = array_merge($extensions, $enum->getExtensions());
                }
            }
        }

        return array_unique($extensions);
    }
}
