<?php

namespace App\Filament\Dashboard\Article;

use Filament\Forms\Form;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\DateTimePicker;
use FilamentTiptapEditor\TiptapEditor;
use FilamentTiptapEditor\Enums\TiptapOutput;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use App\Enums\PostStatus;
use Illuminate\Support\Str;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Hidden;

class ArticleForm
{
    public function getForm($context = 'create')
    {
        $form = [
            Grid::make()
                ->columns([
                    'default' => 1,
                    'lg' => 10,
                ])
                ->schema([
                    // Main Content Column (70%)
                    Group::make()
                        ->columnSpan(['lg' => 7])
                        ->schema([
                            Section::make('Post Details')
                                ->schema([
                                    TextInput::make('post_title')
                                        ->label('Title')
                                        ->placeholder('Enter the post title')
                                        ->required()
                                        ->autofocus()
                                        ->maxLength(191)
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) use ($context) {
                                            if ($context === 'create' && empty($get('slug'))) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->validationMessages([
                                            'regex' => 'Title must only contain letters, numbers, spaces, and &.',
                                        ]),
                                    TextInput::make('slug')
                                        ->label('Slug')
                                        ->placeholder('Enter the post slug')
                                        ->helperText('Will be automatically generated from title if left empty.')
                                        ->maxLength(255)
                                        ->required()
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (?string $state, Set $set) {
                                            if ($state) {
                                                $set('slug', Str::slug($state));
                                            }
                                        }),
                                    Textarea::make('post_excerpt')
                                        ->label('Excerpt')
                                        ->placeholder('Enter a brief excerpt of the post')
                                        ->helperText('A short summary of the post content.')
                                        ->rows(3)
                                        ->maxLength(400),
                                    TiptapEditor::make('post_description')
                                        ->label('Content')
                                        ->placeholder('Enter the post content')
                                        ->profile('dashboard')
                                        ->disk('media')
                                        ->directory('blog')
                                        ->maxSize(2024)
                                        ->output(TiptapOutput::Json)
                                        ->extraInputAttributes(['style' => 'max-height: 50rem; min-height: 20rem;'])
                                        ->required(),
                                ]),
                        ]),

                    // Sidebar Column (30%)
                    Group::make()
                        ->columnSpan(['lg' => 3])
                        ->schema([
                            Section::make('Settings')
                                ->schema([
                                    Hidden::make('post_status')
                                        ->default(PostStatus::Pending->value),
                                    DateTimePicker::make('published_at')
                                        ->label('Published Date')
                                        ->seconds(false)
                                        ->visible(false),
                                    SpatieMediaLibraryFileUpload::make('featured_image')
                                        ->collection('featured_image')
                                        ->imageEditor()
                                        ->reorderable()
                                        ->conversion('thumb'),
                                    Select::make('categories')
                                        ->placeholder('Select categories')
                                        ->relationship('categories', 'name')
                                        ->multiple()
                                        ->preload()
                                        ->searchable(),
                                    Select::make('tags')
                                        ->placeholder('Select tags')
                                        ->relationship('tags', 'name')
                                        ->multiple()
                                        ->preload()
                                        ->searchable(),
                                ]),
                        ]),
                ])
        ];
        return $form;
    }
}
