<?php

namespace App\Filament\Dashboard\Article;

use Filament\Tables\Table;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Illuminate\Support\Str;
use App\Enums\PostStatus;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use App\Models\Blog\Post;
use Illuminate\Support\Facades\Auth;

class ArticleTable
{
    /**
     * Get the table columns.
     *
     * @return array
     */
    public function getTable()
    {
        $table =
            [
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('post_title')
                    ->label('Title')
                    ->tooltip(fn($record): string => $record->post_title)
                    ->wrap()
                    ->searchable()
                    ->formatStateUsing(fn($record): string => Str::limit($record->post_title, 50)),

                TextColumn::make('categories.name')
                    ->wrap(),
                IconColumn::make('is_featured')
                    ->label('Featured'),
                TextColumn::make('post_status')
                    ->badge()
                    ->label('Status'),
                TextColumn::make('created_at')
                    ->label('Date')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->getStateUsing(function ($record): ?string {
                        if (!$record) return null;

                        return $record->post_status->value === PostStatus::Published->value
                            ? $record->published_at
                            : $record->created_at;
                    }),
            ];
        return $table;
    }

    /**
     * Get the table filters.
     *
     * @return array
     */
    public function getFilters()
    {
        $filters = [
            SelectFilter::make('categories')
                ->relationship(
                    name: 'Categories',
                    titleAttribute: 'name',
                    modifyQueryUsing: fn(Builder $query) => $query
                        ->select(['post_categories.id', 'post_categories.name'])
                )
                ->multiple()
                ->preload()
                ->searchable()
                ->optionsLimit(100)
                ->placeholder('All'),
            TernaryFilter::make('is_featured')
                ->label('Featured Posts'),
            SelectFilter::make('created_at')
                ->label('Date')
                ->options(function () {
                    $dates = \App\Models\Blog\Post::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                        ->distinct()
                        ->pluck('month')
                        ->map(function ($date) {
                            return [
                                $date => date('F Y', strtotime($date . '-01'))
                            ];
                        })
                        ->collapse()
                        ->toArray();

                    return $dates;
                })
                ->query(function (Builder $query, array $data): Builder {
                    return $query->when(
                        $data['value'],
                        fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                    );
                }),
        ];
        return $filters;
    }

    /**
     * Get the table actions.
     *
     * @return array
     */
    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('preview')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->label(
                        fn(\App\Models\Blog\Post $record): string =>
                        $record->post_status === PostStatus::Published
                            ? 'View'
                            : 'Preview'
                    )
                    ->url(fn(Post $record): string => route('journal.show', [
                        'slug' => $record->slug,
                        'preview' => $record->post_status !== PostStatus::Published ? true : null,
                    ]), true)
                    ->hidden(
                        fn(Post $record): bool =>
                        !in_array($record->post_status->value, [
                            PostStatus::Published->value,
                            PostStatus::Scheduled->value,
                            PostStatus::Pending->value,
                            PostStatus::Draft->value,
                        ])
                    )
                    ->visible(fn(Post $record): bool => Auth::user()->can('view',['App\Models\Blog\Post',$record->uuid])),

                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->color('info')
                    ->label('Edit')
                    ->url(fn(Post $record): string => route('dashboard.article.edit', ['uuid' => $record->uuid]))
                    ->hidden(fn(Post $record): bool => $record->trashed())
                    ->visible(fn(Post $record): bool => Auth::user()->can('update',['App\Models\Blog\Post',$record->uuid])),
                ReplicateAction::make()
                    ->hidden(fn(Post $record): bool => $record->trashed()),
                DeleteAction::make()
                    ->hidden(fn(Post $record): bool => $record->trashed()),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])->tooltip('Actions'),
        ];
        return $actions;
    }
}
