<?php

namespace App\Filament\Dashboard\Classified;

use Filament\Tables\Table;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Illuminate\Support\Str;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use App\Models\Classified\Classified;
use App\Enums\ClassifiedStatus;
use Illuminate\Support\Facades\Auth;
class ClassifiedTable {

    public function getTable()
    {
        $table =
            [
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('title')
                    ->label('Title')
                    ->tooltip(fn(?Classified $record): string => $record?->title ?? '')
                    ->wrap()
                    ->searchable()
                    ->formatStateUsing(fn(?Classified $record): string => $record ? Str::limit($record->title, 50) : ''),
                TextColumn::make('price')
                    ->money()
                    ->sortable(),
                TextColumn::make('categories.classified_category_name')
                    ->wrap(),
                IconColumn::make('is_featured')
                    ->label('Featured')
                    ->boolean()
                    //->toggleable()
                    ->alignCenter(),
                    TextColumn::make('classified_status')
                    ->badge()
                    ->label('Status'),
                TextColumn::make('city.city_name')
                    ->formatStateUsing(fn(?Classified $record): string => 
                        $record ? collect([
                            $record->city?->city_name,
                            $record->state?->state_name,
                            $record->country?->country_name,
                            $record->postal_code
                        ])->filter()->join(', ') : ''
                    )
                    ->label('Location')
                    ->sortable()
                    ->wrap()
                    ->searchable(),
                 TextColumn::make('published_at')
                    ->label('Date')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->getStateUsing(function (?Classified $record): ?string {
                        if (!$record) return null;
                        
                        return $record->classified_status->value === ClassifiedStatus::Published->value
                            ? $record->published_at
                            : $record->created_at;
                    }),
                TextColumn::make('expired_at')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ];
        return $table;
    }

    public function getFilters()
    {
        $filters = [
        
        TernaryFilter::make('is_featured')
            ->label('Featured Classifieds'),
        SelectFilter::make('classified_categories')
            ->label('Categories')
            ->relationship(
                name: 'Categories',
                titleAttribute: 'classified_category_name',
                modifyQueryUsing: fn(Builder $query) => $query
                    ->select(['classified_categories.id', 'classified_categories.classified_category_name'])
            )
            ->multiple()
            ->preload()
            ->searchable()
            ->optionsLimit(100)
            ->placeholder('All'),
        SelectFilter::make('classified_types')
            ->label('Types')
            ->relationship(
                name: 'Types',
                titleAttribute: 'classified_type_name',
                modifyQueryUsing: fn(Builder $query) => $query
                    ->select(['classified_types.id', 'classified_types.classified_type_name'])
            )
            ->multiple()
            ->preload()
            ->searchable()
            ->optionsLimit(100)
            ->placeholder('All'),
        SelectFilter::make('state_id')
            ->label('State')
            ->relationship('state', 'state_name')
            ->preload()
            ->searchable()
            ->optionsLimit(100)
            ->placeholder('All'),
        SelectFilter::make('city_id')
            ->label('City')
            ->relationship('city', 'city_name')
            ->preload()
            ->searchable()
            ->optionsLimit(100)
            ->placeholder('All'),
        
        SelectFilter::make('created_at')
            ->label('Date')
            ->options(function () {
                $dates = Classified::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                    ->distinct()
                    ->pluck('month')
                    ->map(function ($date) {
                        return [
                            $date => date('F Y', strtotime($date . '-01'))
                        ];
                    })
                    ->collapse()
                    ->toArray();

                return $dates;
            })
            ->query(function (Builder $query, array $data): Builder {
                return $query->when(
                    $data['value'],
                    fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                );
            })
        ];
        return $filters;
    }


    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('preview')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->label(fn (Classified $record): string => 
                        $record->classified_status === ClassifiedStatus::Published 
                        ? 'View' 
                        : 'Preview'
                    )
                    ->url(fn (Classified $record): string => route('classified.show', [
                        'slug' => $record->slug,
                        'preview' => $record->classified_status !== ClassifiedStatus::Published ? true : null,
                        ]), true)
                        ->hidden(fn(Classified $record): bool => 
                        !in_array($record->classified_status->value, [
                            ClassifiedStatus::Published->value,
                            ClassifiedStatus::Scheduled->value,
                            ClassifiedStatus::Pending->value,
                            ClassifiedStatus::Draft->value,
                        ])
                    ),
                
                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->color('info')
                    ->label('Edit')
                    ->url(fn(Classified $record): string => route('dashboard.classified.edit', ['uuid' => $record->uuid]))
                    ->hidden(fn(Classified $record): bool => $record->trashed())
                    ->visible(fn(Classified $record): bool => Auth::user()->can('update',['App\Models\Classified\Classified',$record->uuid])),
                ReplicateAction::make()
                    ->hidden(fn(Classified $record): bool => $record->trashed()),
                DeleteAction::make()
                    ->hidden(fn(Classified $record): bool => $record->trashed()),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])->tooltip('Actions'),
        ];
        return $actions;
    }

}