<?php

namespace App\Filament\Dashboard\Equipment;

use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Forms\Components\DatePicker;
use Illuminate\Support\Str;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Filters\Filter;
use App\Models\Equipment\Equipment;
use App\Enums\EquipmentStatus;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Filters\TrashedFilter;
use Illuminate\Support\Facades\Auth;

class EquipmentTable
{
    /**
     * Get the table columns.
     *
     * @return array
     */
    public function getTable()
    {
        $table = [
            SpatieMediaLibraryImageColumn::make('featured_image')
                ->collection('featured_image')
                ->conversion('thumb')
                ->size(40)
                ->label('Image')
                ->square(),
            TextColumn::make('title')
                ->formatStateUsing(fn($state) => Str::limit($state, 30))
                ->searchable(),
            TextColumn::make('categories.category_name')
                ->sortable()
                ->label('Category')
                ->badge()
                ->limitList(1)
                ->words(3)
                ->wrap()
                ->listWithLineBreaks(),
            TextColumn::make('status')
                ->badge()
                ->formatStateUsing(fn($state) => $state->getLabel())
                ->label('Status'),
            TextColumn::make('address')
                ->wrap()
                ->sortable(),
            TextColumn::make('created_at')
                ->date()
                ->sortable()
                ->label('Created')
        ];

        return $table;
    }

    /**
     * Get the table filters.
     *
     * @return array
     */
    public function getFilters()
    {
        $filters = [
            SelectFilter::make('categories')
                ->relationship('categories', 'category_name')
                ->searchable()
                ->preload()
                ->label('Category'),
            SelectFilter::make('status')
                ->options(EquipmentStatus::class)
                ->label('Status'),
            SelectFilter::make('country')
                ->relationship('country', 'country_name')
                ->searchable()
                ->preload()
                ->label('Country'),
            SelectFilter::make('state')
                ->relationship('state', 'state_name')
                ->searchable()
                ->preload()
                ->label('State'),
            SelectFilter::make('city')
                ->relationship('city', 'city_name')
                ->searchable()
                ->preload()
                ->label('City'),
            Filter::make('created_at')
                ->form([
                    DatePicker::make('created_from')
                        ->label('Created From'),
                    DatePicker::make('created_until')
                        ->label('Created Until'),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when(
                            $data['created_from'],
                            fn(Builder $query, $date): Builder => $query->whereDate('created_at', '>=', $date),
                        )
                        ->when(
                            $data['created_until'],
                            fn(Builder $query, $date): Builder => $query->whereDate('created_at', '<=', $date),
                        );
                })
                ->indicateUsing(function (array $data): array {
                    $indicators = [];

                    if ($data['created_from'] ?? null) {
                        $indicators['created_from'] = 'Created from ' . $data['created_from'];
                    }

                    if ($data['created_until'] ?? null) {
                        $indicators['created_until'] = 'Created until ' . $data['created_until'];
                    }

                    return $indicators;
                }),
            TrashedFilter::make(),
        ];

        return $filters;
    }

    /**
     * Get the table actions.
     *
     * @return array
     */
    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('edit')
                    ->url(fn(Equipment $record): string => route('dashboard.equipments.edit', $record->uuid))
                    ->hidden(fn(Equipment $record): bool => $record->trashed())
                    ->icon('heroicon-o-pencil')
                    ->visible(fn(Equipment $record): bool => Auth::user()->can('update',['App\Models\Equipment\Equipment',$record->uuid])),
                ReplicateAction::make(),
                DeleteAction::make(),
                RestoreAction::make(),
                ForceDeleteAction::make(),
            ])
        ];

        return $actions;
    }

    public function getBulkActions()
    {
        return [
            BulkActionGroup::make([
                DeleteBulkAction::make(),
                ForceDeleteBulkAction::make(),
                RestoreBulkAction::make(),
            ]),
        ];
    }
}
