<?php
namespace App\Filament\Dashboard\Job;

use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Forms\Components\DatePicker;
use Illuminate\Support\Str;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Filters\Filter;
use App\Models\Job\Job;
use App\Enums\JobShift;
use App\Enums\JobSalaryPeriod;
use App\Enums\JobStatus;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class JobTable
{
    /**
     * Get the table columns.
     *
     * @return array
     */
    public function getTable()
    {
        $table =
            [       
            TextColumn::make('job_title')
                ->formatStateUsing(fn ($state) => Str::limit($state, 30))
                ->searchable(),
            
            TextColumn::make('job_role.job_role_name')
                ->sortable(),
            TextColumn::make('job_status')
                ->badge()
                ->label('Status'),
            IconColumn::make('is_featured')
                ->label('Featured'),
            TextColumn::make('salary_period')
                ->searchable(),
            TextColumn::make('job_shift')
                ->searchable(),
            TextColumn::make('apply_by_date')
                ->date()
                ->sortable(),
            TextColumn::make('city.city_name')
                ->sortable(),
            TextColumn::make('dates')
                ->label('Dates')
                ->view('filament.resources.common.tables.columns.dates'),
            ];
        return $table;
    }

    /**
     * Get the table filters.
     *
     * @return array
     */
    public function getFilters()
    {
        $filters = [
            SelectFilter::make('job_role')
                ->relationship('job_role', 'job_role_name')
                ->searchable()
                ->preload()
                ->label('Job Role'),
            SelectFilter::make('salary_period')
                ->options(JobSalaryPeriod::class)
                ->label('Salary Period'),
            SelectFilter::make('job_type')
                ->relationship('job_type', 'job_type_name')
                ->searchable()
                ->preload()
                ->label('Job Type'),
            TernaryFilter::make('is_featured')
                ->label('Featured Status')
                ->placeholder('All Jobs')
                ->trueLabel('Featured Jobs')
                ->falseLabel('Non-Featured Jobs'),
                SelectFilter::make('country')
                ->relationship('country', 'country_name')
                ->searchable()
                ->preload()
                ->label('Country'),
            SelectFilter::make('state')
                ->relationship('state', 'state_name')
                ->searchable()
                ->preload()
                ->label('State'),
            SelectFilter::make('city')
                ->relationship('city', 'city_name')
                ->searchable()
                ->preload()
                ->label('City'),
                Filter::make('published_at')
                    ->form([
                        DatePicker::make('published_from')
                            ->label('Published From'),
                        DatePicker::make('published_until')
                            ->label('Published Until'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['published_from'],
                                fn (Builder $query, $date): Builder => $query->whereDate('published_at', '>=', $date),
                            )
                            ->when(
                                $data['published_until'],
                                fn (Builder $query, $date): Builder => $query->whereDate('published_at', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['published_from'] ?? null) {
                            $indicators['published_from'] = 'Published from ' . Carbon::parse($data['published_from'])->toFormattedDateString();
                        }
                        if ($data['published_until'] ?? null) {
                            $indicators['published_until'] = 'Published until ' . Carbon::parse($data['published_until'])->toFormattedDateString();
                        }
                        return $indicators;
                    }),
                    
        ];
        return $filters;
    }

    /**
     * Get the table actions.
     *
     * @return array
     */
    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('preview')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->label(fn (Job $record): string => 
                        $record->job_status === JobStatus::Published 
                            ? 'View' 
                            : 'Preview'
                    )
                    ->url(fn (Job $record): string => route('job.show', [
                        'slug' => $record->slug,
                        'preview' => $record->job_status !== JobStatus::Published ? true : null,
                    ]), true)
                    ->hidden(fn(Job $record): bool => 
                        !in_array($record->job_status->value, [
                            JobStatus::Published->value,
                            JobStatus::Scheduled->value,
                            JobStatus::Pending->value,
                            JobStatus::Draft->value,
                        ])
                    ),
                
                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->color('info')
                    ->label('Edit')
                    ->url(fn(Job $record): string => route('dashboard.job.edit', ['uuid' => $record->uuid]))
                    ->visible(fn(Job $record): bool => Auth::user()->can('update',['App\Models\Job\Job',$record->uuid]))
                    ->hidden(fn(Job $record): bool => $record->trashed()),

                ReplicateAction::make()
                    ->hidden(fn(Job $record): bool => $record->trashed()),
                DeleteAction::make()
                    ->hidden(fn(Job $record): bool => $record->trashed()),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])->tooltip('Actions'),
        ];
        return $actions;
    }
}

