<?php

namespace App\Filament\Dashboard\Product;

use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Illuminate\Support\Str;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Filament\Traits\HasSeoSection;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Actions\Action;

class ProductForm
{
    use HasSeoSection;
    public function getForm($context = 'create')
    {
        $form = [
            Grid::make()
            ->columns([
                'default' => 1,
                'lg' => 10,
            ])
            ->schema([
                // Main Content Column (70%)
                Group::make()
                    ->columnSpan(['lg' => 7])
                    ->schema([
                        Section::make('Product Details')
                            ->schema([
                                TextInput::make('product_title')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                        if ($context === 'create' && empty($get('slug'))) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (?string $state, Set $set) {
                                        if ($state) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('product_sku')
                                    ->maxLength(255),
                                Textarea::make('product_description')
                                    ->columnSpanFull(),
                                Textarea::make('link')
                                    ->columnSpanFull(),
                            ]),

                        // Gallery Images Section
                        Section::make('Product Images')
                            ->schema([
                                SpatieMediaLibraryFileUpload::make('gallery_images')
                                    ->collection('gallery_images')
                                    ->multiple()
                                    ->reorderable()
                                    ->maxFiles(10)
                                    ->image()
                                    ->imageEditor()
                                    ->columnSpanFull()
                                    ->hint('Upload up to 10 images. Recommended size: 1200x800px')
                                    ->imageResizeMode('cover')
                                    ->imageCropAspectRatio('3:2')
                                    ->imageResizeTargetWidth('1200')
                                    ->imageResizeTargetHeight('800'),
                            ]),
                        // Add SEO Section
                        static::getSeoSection(),
                    ]),

                // Sidebar Column (30%)
                Group::make()
                    ->columnSpan(['lg' => 3])
                    ->schema([
                        Section::make('Settings')
                            ->schema([
                                Select::make('parent_id')
                                    ->relationship('parent', 'product_title')
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent product'),
                                TextInput::make('quantity')
                                    ->required()
                                    ->numeric()
                                    ->default(0),
                                TextInput::make('price')
                                    ->required()
                                    ->numeric()
                                    ->default(0.00)
                                    ->prefix('$'),
                                DateTimePicker::make('published_at')
                                    ->label('Published Date'),
                                DateTimePicker::make('expired_at')
                                    ->label('Expiry Date'),
                                Select::make('categories')
                                    ->multiple()
                                    ->relationship(
                                        name: 'categories',
                                        titleAttribute: 'product_category_name',
                                        modifyQueryUsing: fn(Builder $query) => $query
                                            ->select(['product_categories.id', 'product_categories.product_category_name'])
                                    ),
                            ]),
                    ]),
                ])
            ];
        return $form;
    }
}
