<?php

namespace App\Filament\Dashboard\Project;

use Filament\Tables\Table;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use App\Enums\PostStatus;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Notifications\Notification;
use App\Models\Blog\Post;
use App\Models\Project\Project;
use Filament\Tables\Filters\TrashedFilter;
use App\Enums\ProjectStatus;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class ProjectTable

{
    /**
     * Get the table columns.
     *
     * @return array
     */
    public function getTable()
    {
        $table =
            [
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('title')
                    ->searchable()
                    ->tooltip(fn($state) => $state)
                    ->formatStateUsing(fn($state) => Str::limit($state, 20)),
                TextColumn::make('types.type_name')
                    ->label('Type')
                    ->wrap(),
                TextColumn::make('project_status')
                    ->label('Status')
                    ->badge()
                    ->searchable(),
                TextColumn::make('user.name')
                    ->label('Belongs To')
                    ->sortable(),
                TextColumn::make('published_at')
                    ->dateTime()
                    ->sortable(),
                TextColumn::make('expired_at')
                    ->dateTime()
                    ->sortable(),
            ];
        return $table;
    }

    /**
     * Get the table filters.
     *
     * @return array
     */
    public function getFilters()
    {
        $filters = [
            TrashedFilter::make(),
            SelectFilter::make('project_status')
                ->options([
                    ProjectStatus::Draft->value => ProjectStatus::Draft->name,
                    ProjectStatus::Published->value => ProjectStatus::Published->name,
                    ProjectStatus::Pending->value => ProjectStatus::Pending->name,
                    ProjectStatus::Expired->value => ProjectStatus::Expired->name,
                    ProjectStatus::Trashed->value => ProjectStatus::Trashed->name,
                ])
                ->label('Status'),
        ];
        return $filters;
    }

    /**
     * Get the table actions.
     *
     * @return array
     */
    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('view')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->label('View')
                    ->url(fn(Project $record): string => route('project.show', ['slug' => $record->slug]))
                    ->extraAttributes(['target' => '_blank'])
                    ->hidden(fn(Project $record): bool => $record->trashed()),
                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->color('info')
                    ->label('Edit')
                    ->url(fn(Project $record): string => route('dashboard.project.edit', ['uuid' => $record->uuid]))
                    ->hidden(fn(Project $record): bool => $record->trashed())
                    ->visible(fn(Project $record): bool => Auth::user()->can('update',['App\Models\Project\Project',$record->uuid])),
                ReplicateAction::make()
                    ->hidden(fn(Project $record): bool => $record->trashed()),
                DeleteAction::make()
                    ->hidden(fn(Project $record): bool => $record->trashed()),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])->tooltip('Actions'),
        ];
        return $actions;
    }
}
