<?php

namespace App\Filament\Dashboard\Team;

use Filament\Forms\Form;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Hidden;
use App\Forms\Components\TinyMceEditor;

class TeamForm
{
    public function getForm($context = 'create')
    {
        $form = [
            Grid::make()
                ->columns([
                    'default' => 1,
                    'lg' => 10,
                ])
                ->schema([
                    // Main Content Column (70%)
                    Group::make()
                        ->columnSpan(['lg' => 7])
                        ->schema([
                            Section::make('Basic Information')
                                ->description('Enter the team member\'s basic details')
                                ->schema([
                                    TextInput::make('first_name')
                                        ->placeholder('Enter the team member\'s first name')
                                        ->required()
                                        ->maxLength(100),

                                    TextInput::make('last_name')
                                        ->placeholder('Enter the team member\'s last name')
                                        ->maxLength(100),
                                    TextInput::make('email')
                                        ->email()
                                        ->placeholder('Enter the team member\'s email')
                                        ->maxLength(100),
                                    TextInput::make('phone')
                                        ->tel()
                                        ->mask('(999) 999-9999')
                                        ->placeholder('Enter the team member\'s phone number')
                                        ->maxLength(100),
                                    TextInput::make('experience_years')
                                        ->placeholder('Enter the team member\'s experience years')
                                        ->maxLength(100)
                                        ->regex('/^[0-9]+$/'),
                                    TextInput::make('team_member_role')
                                        ->placeholder('Enter the team member\'s role')
                                        ->helperText('Enter the team member\'s role (max 30 characters)')
                                        ->maxLength(30),
                                ])->columns(2),

                            Section::make('Professional Details')
                                ->schema([
                                    Repeater::make('credentials')
                                        ->label('Credentials & Certifications')
                                        ->schema([
                                            TextInput::make('title')
                                                ->label('Credential Title')
                                                ->required()
                                                ->maxLength(20)
                                                ->placeholder('e.g., Water Treatment Operator'),
                                            TextInput::make('description')
                                                ->label('Description')
                                                ->required()
                                                ->maxLength(100)
                                                ->placeholder('e.g., Certified in 2023 by ABC Organization'),
                                        ])
                                        ->columns(1)
                                        ->defaultItems(0)
                                        ->addActionLabel('Add Credential')
                                        ->reorderableWithButtons()
                                        ->collapsible()
                                        ->itemLabel(fn(array $state): ?string => $state['title'] ?? null)
                                        ->columnSpanFull(),
                                    TinyMceEditor::make('description')
                                        ->plugins('lists link autolink autoresize')
                                        ->toolbar('bold italic underline | bullist numlist | undo redo')
                                        ->menubar(false)
                                        ->branding(false)
                                        ->promotion(false)
                                        ->height(300)
                                        ->placeholder('Enter the team member\'s description')
                                        ->columnSpanFull(),
                                ]),
                        ]),

                    // Sidebar Column (30%)
                    Group::make()
                        ->columnSpan(['lg' => 3])
                        ->schema([

                            Section::make('Media')
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('featured_image')
                                        ->collection('featured_image')
                                        ->imageEditor()
                                        ->hintIcon('heroicon-m-photo')
                                        ->conversion('thumb')
                                        ->required(),
                                    Hidden::make('team_member_status'),
                                ]),

                            Section::make('Contact & Social')
                                ->schema([
                                    Textarea::make('address')
                                        ->placeholder('Enter the team member\'s address')
                                        ->rows(3),
                                    TextInput::make('social_links.facebook')
                                        ->label('Facebook')
                                        ->url()
                                        ->prefixIcon('fab-facebook')
                                        ->placeholder('https://facebook.com/username'),
                                    TextInput::make('social_links.linkedin')
                                        ->label('LinkedIn')
                                        ->url()
                                        ->prefixIcon('fab-linkedin')
                                        ->placeholder('https://linkedin.com/in/username'),
                                    TextInput::make('social_links.youtube')
                                        ->label('YouTube')
                                        ->url()
                                        ->prefixIcon('fab-youtube')
                                        ->placeholder('https://youtube.com/@username'),
                                    TextInput::make('social_links.instagram')
                                        ->label('Instagram')
                                        ->url()
                                        ->prefixIcon('fab-instagram')
                                        ->placeholder('https://instagram.com/username'),
                                    TextInput::make('social_links.x')
                                        ->label('X (Twitter)')
                                        ->url()
                                        ->prefixIcon('fab-x-twitter')
                                        ->placeholder('https://x.com/username'),
                                ]),
                        ]),
                ])
        ];
        return $form;
    }
}
