<?php
namespace App\Filament\Dashboard\Video;

use App\Models\User\UserVideo;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Filters\TrashedFilter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class VideoTable
{
    public function getTable()
    {
        $table = [
            SpatieMediaLibraryImageColumn::make('video')
                ->collection('video')
                ->conversion('thumb'),
            TextColumn::make('video_title')
                ->label('Title')
                ->tooltip(fn(?UserVideo $record): string => $record?->video_title ?? '')
                ->wrap()
                ->searchable()
                ->sortable()
                ->formatStateUsing(fn(?UserVideo $record): string => $record ? Str::limit($record->video_title, 50) : ''),

            TextColumn::make('video_link')
                ->label('Video Link')
                ->limit(30)
                ->tooltip(fn(?UserVideo $record): string => $record?->video_link ?? '')
                ->url(fn(UserVideo $record): ?string => $record->video_link ?? null)
                ->openUrlInNewTab(),

            IconColumn::make('is_active')
                ->label('Status')
                ->boolean()
                ->trueIcon('heroicon-o-check-circle')
                ->falseIcon('heroicon-o-x-circle')
                ->trueColor('success')
                ->falseColor('danger')
                ->alignCenter(),
            IconColumn::make('is_featured')
                ->boolean()
                ->trueIcon('heroicon-s-star')
                ->falseIcon('heroicon-o-star')
                ->trueColor('warning')
                ->falseColor('gray')
                ->label('Featured'),

            TextColumn::make('user.name')
                ->label('Owner')
                ->searchable()
                ->sortable(),

            TextColumn::make('created_at')
                ->label('Created At')
                ->dateTime()
                ->sortable(),

            TextColumn::make('updated_at')
                ->label('Updated At')
                ->dateTime()
                ->sortable()
                ->toggleable(isToggledHiddenByDefault: true),
        ];

        return $table;
    }

    public function getFilters()
    {
        $filters = [
            TernaryFilter::make('is_active')
                ->falseLabel('Inactive')
                ->trueLabel('Active')
                ->placeholder('All videos')
                ->label('Active Videos'),

            SelectFilter::make('created_at')
                ->label('Date')
                ->options(function () {
                    $dates = UserVideo::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                        ->distinct()
                        ->pluck('month')
                        ->map(function ($date) {
                            return [
                                $date => date('F Y', strtotime($date . '-01')),
                            ];
                        })
                        ->collapse()
                        ->toArray();

                    return $dates;
                })
                ->query(function (Builder $query, array $data): Builder {
                    return $query->when(
                        $data['value'],
                        fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                    );
                }),

            TrashedFilter::make(),

        ];

        return $filters;
    }

    public function getActions()
    {
        $actions = [
            ActionGroup::make([
                Action::make('view')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->label('View')
                    ->url(fn(UserVideo $record): string => $record->video_link ?? '#', true)
                    ->hidden(fn(UserVideo $record): bool => empty($record->video_link) || $record->trashed()),

                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->color('warning')
                    ->label('Edit')
                    ->url(fn(UserVideo $record): string => route('dashboard.video.edit', ['uuid' => $record->uuid]))
                    ->hidden(fn(UserVideo $record): bool => $record->trashed())
                    ->visible(fn(UserVideo $record): bool => Auth::user()->can('update', ['App\Models\User\UserVideo', $record->uuid])),
                Action::make('feature')
                    ->icon('heroicon-o-star')
                    ->action(function (UserVideo $record) {
                        // First, unfeature any currently featured video
                        if (! $record->is_featured) {
                            UserVideo::where('is_featured', true)->update(['is_featured' => false]);
                            $record->update(['is_featured' => true]);
                        }
                    })
                    ->color(fn(UserVideo $record) => $record->is_featured ? 'warning' : 'gray')
                    ->icon(fn(UserVideo $record) => $record->is_featured ? 'heroicon-s-star' : 'heroicon-o-star')
                    ->tooltip(fn(UserVideo $record) => $record->is_featured ? 'This video is currently featured' : 'Mark as featured')
                    ->hidden(fn(UserVideo $record) => $record->is_featured),
                ReplicateAction::make()
                    ->hidden(fn(UserVideo $record): bool => $record->trashed()),

                DeleteAction::make()
                    ->hidden(fn(UserVideo $record): bool => $record->trashed()),

                ForceDeleteAction::make(),

                RestoreAction::make(),
            ])->tooltip('Actions'),
        ];

        return $actions;
    }

    public function getBulkActions()
    {
        return [
            BulkActionGroup::make([
                DeleteBulkAction::make(),
                RestoreBulkAction::make(),
                ForceDeleteBulkAction::make(),
            ]),
        ];
    }
}
