<?php

namespace App\Filament\Imports\Blog;

use App\Models\Blog\Post;
use App\Models\Blog\PostCategory;
use Filament\Actions\Imports\ImportColumn;
use Filament\Actions\Imports\Importer;
use Filament\Actions\Imports\Models\Import;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class PostImporter extends Importer
{
    protected static ?string $label = 'Import From CSV';

    protected static ?string $model = Post::class;

    public static function getColumns(): array
    {
        return [
            ImportColumn::make('post_title')
                ->requiredMapping()
                ->rules(['required', 'max:255']),
            ImportColumn::make('post_description'),
            ImportColumn::make('slug')
                ->helperText('Leave blank to auto-generate.'),
            ImportColumn::make('category_name')
                ->requiredMapping()
                ->rules(['required', 'string', 'max:255']),
        ];
    }

    public function resolveRecord(): ?Post
    {
        try {
            Log::channel('daily')->info('Import started', [
                'data' => $this->data,
                'options' => $this->options
            ]);

            // Find or create category
            $category = PostCategory::firstOrCreate(
                ['name' => $this->data['category_name']],
                ['slug' => Str::slug($this->data['category_name'])]
            );

            Log::channel('daily')->info('Category processed', [
                'category' => $category->toArray()
            ]);

            // Set post status from default option
            $this->data['post_status'] = $this->options['default_status'] ?? 'draft';

            // Set published_at if status is published
            $published_at = null;
            if ($this->data['post_status'] === 'published') {
                $published_at = now();
            }

            if ($this->options['updateExisting'] ?? false) {
                $post = Post::firstOrNew([
                    'post_title' => $this->data['post_title'],
                ]);
            } else {
                $post = new Post();
            }

            // Mutate the record with all the data
            $post->forceFill([
                'post_title' => $this->data['post_title'],
                'post_description' => $this->data['post_description'] ?? '',
                'slug' => $this->data['slug'] ?? Str::slug($this->data['post_title']),
                'post_status' => $this->data['post_status'],
                'created_by' => auth()->user()?->id ?? 1, // Fallback to ID 1 if no auth
                'is_featured' => false, // Default to false for imported posts
                'published_at' => $published_at,
            ]);

            // Save the post to get an ID
            $post->save();

            // Attach the category
            $post->categories()->sync([$category->id]);

            Log::channel('daily')->info('Post processed', [
                'post' => $post->toArray()
            ]);
            
            return $post;
        } catch (\Exception $e) {
            Log::error('Error importing post: ' . $e->getMessage(), [
                'post_title' => $this->data['post_title'] ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            throw $e;
        }
    }

    public static function getCompletedNotificationBody(Import $import): string
    {
        $body = 'Your post import has completed and ' . number_format($import->successful_rows) . ' ' . str('row')->plural($import->successful_rows) . ' imported.';

        if ($failedRowsCount = $import->getFailedRowsCount()) {
            $body .= ' ' . number_format($failedRowsCount) . ' ' . str('row')->plural($failedRowsCount) . ' failed to import.';
        }

        return $body;
    }

    public static function getOptionsFormComponents(): array
    {
        return [
            Checkbox::make('updateExisting')
                ->label('Update existing records'),
            Select::make('default_status')
                ->label('Default Post Status')
                ->options([
                    'published' => 'Published',
                    'draft' => 'Draft',
                    'pending' => 'Pending',
                ])
                ->helperText('All imported posts will use this status')
                ->required()
                ->default('draft'),
        ];
    }

    public function mutateBeforeCreate(array $data): array
    {
        Log::channel('daily')->info('Mutating data before create', ['data' => $data]);
        return $data;
    }
}
