<?php

namespace App\Filament\PageTemplates\Widgets;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
 
use App\Enums\WidgetComponentType;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Grid;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Models\Classified\Classified;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Location\City;
use App\Models\Classified\ClassifiedType;
use App\Models\Classified\ClassifiedCategory;
use App\Enums\ClassifiedCondition;

class ClassifiedList extends Widget
{
    public static ?string $view = 'frontend.widgets.classified-list';
    public static ?string $componentType = WidgetComponentType::Livewire->value;
    public static function title(): string
    {
        return 'Classified List';
    }



    public static function schema(): array
    {
        return [
            RichEditor::make('heading')
                ->required()
                ->label('Heading')
                ->toolbarButtons([
                    'bold',
                    'italic',
                    'underline',
                    'bulletList',
                    'orderedList',
                    'link',
                ]),
            RichEditor::make('description')
                ->label('Description')
                ->toolbarButtons([
                    'bold',
                    'italic',
                    'underline',
                    'bulletList',
                    'orderedList',
                    'link',
                ]),
            Toggle::make('selected_classifieds')
                ->label('Selected Classifieds')
                ->default(true)
                ->reactive()
                ->afterStateUpdated(function (Set $set, Get $get, ?bool $state) {
                    if ($state) {
                        $set('latest_classifieds', false);
                        return;
                    }
                    if (!(bool) $get('latest_classifieds')) {
                        $set('latest_classifieds', true);
                    }
                }),
            Toggle::make('latest_classifieds')
                ->label('Latest Classifieds')
                ->default(false)
                ->reactive()
                ->afterStateUpdated(function (Set $set, Get $get, ?bool $state) {
                    if ($state) {
                        $set('selected_classifieds', false);
                        return;
                    }
                    if (!(bool) $get('selected_classifieds')) {
                        $set('selected_classifieds', true);
                    }
                }),

            // Selected classifieds picker
            Select::make('classifieds')
                ->label('Classifieds')
                ->options(fn () => Classified::query()->orderBy('title')->pluck('title', 'id'))
                ->searchable()
                ->preload()
                ->multiple()
                ->visible(fn (Get $get): bool => (bool) $get('selected_classifieds')),

            // Latest classifieds filter fields
            Grid::make(4)
                ->schema([
                    Select::make('country_id')
                        ->label('Country')
                        ->options(fn () => Country::query()->orderBy('country_name')->pluck('country_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),
                    Select::make('state_id')
                        ->label('State')
                        ->options(fn (Get $get) => $get('country_id') ? State::query()
                            ->where('country_id', $get('country_id'))
                            ->orderBy('state_name')
                            ->pluck('state_name', 'id') : [])
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn (Get $get): bool => empty($get('country_id')))
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),
                    Select::make('city_id')
                        ->label('City')
                        ->options(fn (Get $get) => $get('state_id') ? City::query()
                            ->where('state_id', $get('state_id'))
                            ->orderBy('city_name')
                            ->pluck('city_name', 'id') : [])
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn (Get $get): bool => empty($get('state_id')))
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    Select::make('type')
                        ->label('Ad Type')
                        ->multiple()
                        ->options(fn () => ClassifiedType::query()->orderBy('classified_type_name')->pluck('classified_type_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    Select::make('category')
                        ->label('Category')
                        ->multiple()
                        ->options(fn () => ClassifiedCategory::query()
                            ->whereNull('parent_id')
                            ->orderBy('classified_category_name')
                            ->pluck('classified_category_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),
                    Select::make('sub_category')
                        ->label('Sub Category')
                        ->multiple()
                        ->options(function (Get $get) {
                            $categoryIds = (array) ($get('category') ?? []);
                            if (empty($categoryIds)) {
                                return [];
                            }
                            return ClassifiedCategory::query()
                                ->whereIn('parent_id', $categoryIds)
                                ->orderBy('classified_category_name')
                                ->pluck('classified_category_name', 'id');
                        })
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn (Get $get): bool => empty($get('category')))
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    Select::make('condition')
                        ->label('Condition')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->options(function () {
                            return collect(ClassifiedCondition::cases())
                                ->mapWithKeys(fn (ClassifiedCondition $c) => [$c->value => $c->getLabel()])
                                ->all();
                        })
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    Select::make('item_age')
                        ->label('Item Age')
                        ->searchable()
                        ->preload()
                        ->options([
                            'new' => 'New (<= 1 month)',
                            '1-3' => '1-3 years',
                            '3-5' => '3-5 years',
                            '5-10' => '5-10 years',
                            '10-20' => '10-20 years',
                            '100+' => '100+ years',
                        ])
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    Select::make('date_posted')
                        ->label('Date Posted')
                        ->searchable()
                        ->preload()
                        ->options([
                            '24_hours' => 'Last 24 hours',
                            '7_days' => 'Last 7 days',
                            '30_days' => 'Last 30 days',
                            'any' => 'Any time',
                        ])
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),

                    TextInput::make('limit')
                        ->label('Limit')
                        ->numeric()
                        ->default(3)
                        ->visible(fn (Get $get): bool => (bool) $get('latest_classifieds')),
                ]),
        ];
    }

    public static function hiddenOnTemplateOptions(): bool
    {
        return false;
    }
}
