<?php

namespace App\Filament\PageTemplates\Widgets;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use App\Enums\WidgetComponentType;
use Filament\Forms\Components\Select;
use App\Models\Job\Job;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Location\City;
use App\Models\Job\JobRole;
use App\Models\Job\JobType;
use App\Models\Job\JobCareerLevel;
use App\Enums\JobShift;
use App\Enums\JobWorkMode;
   
use Filament\Forms\Components\Grid;

class JobList extends Widget
{
    public static ?string $view = 'frontend.widgets.job-list';
    public static ?string $componentType = WidgetComponentType::Livewire->value;
    public static function title(): string
    {
        return 'Job List';
    }



    public static function schema(): array
    {
        return [
            RichEditor::make('heading')
                ->required()
                ->label('Heading')
                ->toolbarButtons([
                    'bold',
                    'italic',
                    'underline',
                    'bulletList',
                    'orderedList',
                    'link',
                ]),
            RichEditor::make('description')
                ->label('Description')
                ->toolbarButtons([
                    'bold',
                    'italic',
                    'underline',
                    'bulletList',
                    'orderedList',
                    'link',
                ]),
            Toggle::make('selected_jobs')
                ->label('Selected Jobs')
                ->default(true)
                ->reactive()
                ->afterStateUpdated(function (Set $set, Get $get, ?bool $state) {
                    if ($state) {
                        $set('latest_jobs', false);
                        return;
                    }
                    if (! (bool) $get('latest_jobs')) {
                        $set('latest_jobs', true);
                    }
                }),
            Toggle::make('latest_jobs')
                ->label('Latest Jobs')
                ->default(false)
                ->reactive()
                ->afterStateUpdated(function (Set $set, Get $get, ?bool $state) {
                    if ($state) {
                        $set('selected_jobs', false);
                        return;
                    }
                    if (! (bool) $get('selected_jobs')) {
                        $set('selected_jobs', true);
                    }
                }),
            Select::make('jobs')
                ->label('Jobs')
                ->options(Job::all()->pluck('job_title', 'id'))
                ->searchable()
                ->preload()
                ->multiple()
                ->visible(fn(Get $get): bool => (bool) $get('selected_jobs')),

            // Latest jobs filter fields
            Grid::make(4)
                ->schema([
                    Select::make('country_id')
                        ->label('Country')
                        ->options(fn() => Country::query()->orderBy('country_name')->pluck('country_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),
                    Select::make('state_id')
                        ->label('State')
                        ->options(fn(Get $get) => $get('country_id') ? State::query()
                            ->where('country_id', $get('country_id'))
                            ->orderBy('state_name')
                            ->pluck('state_name', 'id') : [])
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn(Get $get): bool => empty($get('country_id')))
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),
                    Select::make('city_id')
                        ->label('City')
                        ->options(fn(Get $get) => $get('state_id') ? City::query()
                            ->where('state_id', $get('state_id'))
                            ->orderBy('city_name')
                            ->pluck('city_name', 'id') : [])
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn(Get $get): bool => empty($get('state_id')))
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    Select::make('job_role')
                        ->label('Role')
                        ->options(fn() => JobRole::query()->orderBy('job_role_name')->pluck('job_role_name', 'id'))
                        ->searchable()
                        ->multiple()
                        ->preload()
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    Select::make('job_type_id')
                        ->label('Job Type')
                        ->options(fn() => JobType::query()->orderBy('job_type_name')->pluck('job_type_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    Select::make('job_shift')
                        ->label('Job Shift')
                        ->searchable()
                        ->preload()
                        ->options(function () {
                            return collect(JobShift::cases())
                                ->mapWithKeys(fn(JobShift $shift) => [$shift->value => $shift->getLabel()])
                                ->all();
                        })
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    Select::make('job_career_level_id')
                        ->label('Experience')
                        ->options(fn() => JobCareerLevel::query()->orderBy('career_level_name')->pluck('career_level_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    

                    Select::make('work_mode')
                        ->label('Remote / Work Mode')
                        ->options(function () {
                            return collect(JobWorkMode::cases())
                                ->mapWithKeys(fn(JobWorkMode $mode) => [$mode->value => $mode->getLabel()])
                                ->all();
                        })
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),


                    DatePicker::make('apply_by_date')
                        ->label('Deadline')
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),

                    TextInput::make('limit')
                        ->label('Limit')
                        ->numeric()
                        ->default(3)
                        ->visible(fn(Get $get): bool => (bool) $get('latest_jobs')),
                ]),
        ];
    }

    public static function hiddenOnTemplateOptions(): bool
    {
        return false;
    }
}
