<?php

namespace App\Filament\PageTemplates\Widgets;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use App\Enums\WidgetComponentType;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use App\Models\User\User;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Radio;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Models\User\MemberCategory;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Location\City;
 
class TopMembers extends Widget
{
    public static ?string $view = 'frontend.widgets.top-members';
    public static ?string $componentType = WidgetComponentType::Livewire->value;

    public static function title(): string
    {
        return 'Top Members';
    }



    public static function schema(): array
    {
        return [
            Grid::make()
                ->columns(12)
                ->schema([
                    RichEditor::make('heading')
                        ->required()
                        ->label('Heading')
                        ->toolbarButtons([
                            'bold',
                            'italic',
                            'underline',
                            'bulletList',
                            'orderedList',
                            'link',
                        ])
                        ->columnSpan(12),
                    RichEditor::make('description')
                        ->label('Description')
                        ->toolbarButtons([
                            'bold',
                            'italic',
                            'underline',
                            'bulletList',
                            'orderedList',
                            'link',
                        ])
                        ->columnSpan(12),
                    Radio::make('member_selection_type')
                        ->label('Member Selection Type')
                        ->options([
                            'use_selected_members' => 'Use Selected Members',
                            'use_latest_members' => 'Use Latest Members',
                        ])
                        ->default('use_latest_members')
                        ->reactive()
                        ->required()
                        ->inline()
                        ->columnSpan(12),
                    Select::make('members')
                        ->label('Members')
                        ->multiple()
                        ->options(User::role('Service')->pluck('name', 'id'))
                        ->searchable()
                        ->preload()
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_selected_members')
                        ->columnSpan(12),

                    Select::make('categories')
                        ->label('Categories')
                        ->multiple()
                        ->options(fn () => MemberCategory::query()
                            ->orderBy('category_name')
                            ->pluck('category_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('subcategories')
                        ->label('Subcategories')
                        ->multiple()
                        ->options(function (Get $get) {
                            $categoryIds = $get('categories');
                            if (!$categoryIds) {
                                return [];
                            }
                            return MemberCategory::query()
                                ->whereIn('parent_id', $categoryIds)
                                ->orderBy('category_name')
                                ->pluck('category_name', 'id');
                        })
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->disabled(fn (Get $get): bool => blank($get('categories')))
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('services')
                        ->label('Services')
                        ->multiple() 
                        ->options(function (Get $get) {
                            $subcategoryIds = $get('subcategories');
                            if (!$subcategoryIds) {
                                return [];
                            }
                            return MemberCategory::query()
                                ->whereIn('parent_id', $subcategoryIds)
                                ->orderBy('category_name')
                                ->pluck('category_name', 'id');
                        })
                        ->searchable()
                        ->preload()
                        ->disabled(fn (Get $get): bool => blank($get('subcategories')))
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('country_id')
                        ->label('Country')
                        ->options(fn () => Country::query()
                            ->orderBy('country_name')
                            ->pluck('country_name', 'id'))
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->afterStateUpdated(function (Set $set): void {
                            $set('state_id', null);
                            $set('city_id', null);
                        })
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('state_id')
                        ->label('State')
                        ->options(function (Get $get) {
                            $countryId = $get('country_id');
                            if (!$countryId) {
                                return [];
                            }
                            return State::query()
                                ->where('country_id', $countryId)
                                ->orderBy('state_name')
                                ->pluck('state_name', 'id');
                        })
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->afterStateUpdated(function (Set $set): void {
                            $set('city_id', null);
                        })
                        ->disabled(fn (Get $get): bool => blank($get('country_id')))
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('city_id')
                        ->label('City')
                        ->options(function (Get $get) {
                            $stateId = $get('state_id');
                            $countryId = $get('country_id');
                            $query = City::query();
                            if ($stateId) {
                                $query->where('state_id', $stateId);
                            } elseif ($countryId) {
                                $query->where('country_id', $countryId);
                            } else {
                                return [];
                            }
                            return $query->orderBy('city_name')->pluck('city_name', 'id');
                        })
                        ->searchable()
                        ->preload()
                        ->disabled(fn (Get $get): bool => blank($get('state_id')) && blank($get('country_id')))
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Select::make('rating')
                        ->label('Member Ratings')
                        ->options([
                            1 => '1 star+',
                            2 => '2 stars+',
                            3 => '3 stars+',
                            4 => '4 stars+',
                            5 => '5 stars+',
                        ])
                        ->native(false)
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(6),
                        TextInput::make('limit')
                        ->label('Limit')
                        ->numeric()
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(6),

                    Toggle::make('with_reviews')
                        ->label('Only With Reviews')
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                    Toggle::make('emergency_only')
                        ->label('Emergency Services Only')
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),
                    Toggle::make('is_featured')
                        ->label('Featured Members')
                        ->hidden(fn (Get $get): bool => $get('member_selection_type') !== 'use_latest_members')
                        ->columnSpan(4),

                ]),
        ];
    }

    public static function hiddenOnTemplateOptions(): bool
    {
        return false;
    }
}
